import { connect } from 'react-redux';
import { reduxForm, formValueSelector } from 'redux-form';
import createI18N from '@yandex-int/i18n';
import * as commonKeyset from 'common.i18n';
import * as keyset from '../delegation.i18n';
import * as actions from '../actions';
import { ROUTE } from '../constants';
import DelegationForm from '../components/DelegationForm';

const i18n = createI18N(keyset);
const i18nRequired = createI18N(commonKeyset)('inputRequired');
const i18nFromError = i18n('fromError');
const i18nToError = i18n('toError');
const i18nFromToError = i18n('fromToError');

const validate = (values, props) => {
  const errors = {};
  const requiredFields = ['from', 'to'];

  requiredFields.forEach(field => {
    if (!values[field] || (values[field].hasOwnProperty('length') && !values[field].length)) {
      errors[field] = i18nRequired;
    }
  });

  if (!values.executorId) {
    errors.executor = i18nRequired;
  }

  if (values.from) {
    const from = new Date(values.from);
    const today = new Date();

    if (
      from !== 'Invalid Date' &&
      values.from.length > 9 &&
      from.setHours(0, 0, 0, 0) < today.setHours(0, 0, 0, 0) &&
      props.initialValues.canEditStartDate !== false
    ) {
      errors.from = i18nFromError;
    }
  }

  if (values.to) {
    const to = new Date(values.to);
    const today = new Date();

    if (
      to !== 'Invalid Date' &&
      values.to.length > 9 &&
      to.setHours(0, 0, 0, 0) < today.setHours(0, 0, 0, 0) &&
      props.initialValues.canEditEndDate !== false
    ) {
      errors.to = i18nToError;
    }
  }

  if (values.from && values.to && new Date(values.from) > new Date(values.to)) {
    errors.to = i18nFromToError;
  }

  return errors;
};

const mapState = (state, props) => {
  const formName = props.form;
  const selector = formValueSelector(formName);
  const { delegationId, executorName, to, canEditEndDate, canEditStartDate } = selector(
    state,
    'delegationId',
    'executorName',
    'to',
    'canEditEndDate',
    'canEditStartDate',
  );

  return {
    delegationId,
    executorName,
    to,
    canEditEndDate,
    canEditStartDate,
  };
};

const mapDispatch = (dispatch, props) => {
  const loadData = () => dispatch(actions.loadData());

  return {
    onSubmit: data => {
      const { executor, user, ...sendData } = data;

      dispatch(actions.saveDelegation(sendData)).then(responce => {
        if (responce.success === true) {
          loadData().then(() => {
            props.router.push(ROUTE);
          });
        }

        return responce;
      });
    },
    onDelete: () => {
      const { executorName, delegationId } = props.initialValues;
      const result = confirm(
        i18n('removeConfirm', {
          colleague: executorName,
        }),
      );

      if (!result) return;

      dispatch(actions.deleteDelegation(delegationId)).then(responce => {
        if (responce.success === true) {
          loadData().then(() => {
            props.router.push(ROUTE);
          });
        }

        return responce;
      });
    },
    onReset: () => {
      dispatch(actions.resetDelegation());
      props.router.push(ROUTE);
    },
  };
};

export default connect(mapState, mapDispatch)(reduxForm({ validate })(DelegationForm));
