import React from 'react';
import PropTypes from 'prop-types';
import { Field } from 'redux-form';
import TextInput from 'lego/redux-form/TextInput';
import Button from '@crm/components/dist/lego2/Button';
import Toolbar from 'components/Toolbar';
import cx from 'classnames';
import createI18N, { i18nRaw as createI18NRaw } from '@yandex-int/i18n';
import * as commonKeyset from 'common.i18n';
import * as keyset from './FolderForm.i18n';
import FolderSelect from '../../modules/folderSelect';
import css from './styles.modules.scss';

const commonI18n = createI18N(commonKeyset);
const i18n = createI18N(keyset);
const i18nFolderName = i18n('folderName');
const i18nNewFolder = i18n('newFolder');
const i18nMoveToFolder = i18n('moveToFolder');
const i18nSave = commonI18n('save');
const i18nCancel = commonI18n('cancel');
const i18nRemove = commonI18n('remove');
const i18nHint = createI18NRaw(keyset)('hint', {
  br: <br />,
});

const SettingsFolderForm = (props) => {
  const {
    handleSubmit,
    submitting,
    invalid,
    name,
    counter: { total },
    tree,
    onReset,
    onDelete,
  } = props;
  const { id } = props.initialValues;
  const canDelete = total || (tree[id] && tree[id].items && tree[id].items.length);

  return (
    <form onSubmit={handleSubmit} className={css.form}>
      <div className={css.block}>
        <div className={css.header}>
          <label htmlFor="color" className={cx(css.label, css.requiredStar)}>
            {i18nFolderName}
          </label>
          {!id ? <span className={css.isNew}>{i18nNewFolder}</span> : null}
        </div>
        <Field
          placeholder={i18nFolderName}
          name="name"
          id="name"
          className={css.input}
          component={TextInput}
        />
      </div>
      <div className={css.block}>
        <label htmlFor="parentId" className={css.label}>
          {i18nMoveToFolder}
        </label>
        <Field
          name="parentId"
          id="parentId"
          className={css.input}
          component={FolderSelect}
          props={{ id, tree }}
        />
      </div>

      <div className={css.actions}>
        <Toolbar>
          <Button type="submit" size="s" view="action" disabled={submitting || invalid}>
            {i18nSave}
          </Button>
          <Button size="s" view="pseudo" disabled={submitting} onClick={onReset}>
            {i18nCancel}
          </Button>
        </Toolbar>
        {id ? (
          <span style={{ textAlign: 'right' }}>
            <Button
              size="s"
              view="pseudo"
              disabled={submitting || canDelete}
              onClick={() => {
                onDelete(name);
              }}
            >
              {i18nRemove}
            </Button>
            {canDelete ? <span className={css.hint}>{i18nHint}</span> : null}
          </span>
        ) : null}
      </div>
    </form>
  );
};

SettingsFolderForm.propTypes = {
  id: PropTypes.number,
  counter: PropTypes.shape({
    total: PropTypes.number,
  }),
  tree: PropTypes.instanceOf(Object),
  onSubmit: PropTypes.func.isRequired,
  onDelete: PropTypes.func,
  onReset: PropTypes.func,
};

SettingsFolderForm.defaultProps = {
  id: undefined,
  counter: { total: 0 },
  tree: {},
  onDelete: () => {},
  onReset: () => {},
};

export default SettingsFolderForm;
