import React from 'react';
import PropTypes from 'prop-types';
import { fromGrantAdapter } from 'lego/components/SuggestStaff';
import cx from 'classnames';
import createI18N from '@yandex-int/i18n';
import { ConnectedTagsForm as TagForm } from 'modules/tags/modules/tagForm';
import * as keyset from './TagsLayout.i18n';
import TagsListWithFilter from './TagsListWithFilter';
import Tip from '../Tips/Tags';
import css from './styles.modules.scss';

const i18n = createI18N(keyset);
const i18nTitle = i18n('title');
const i18nNewTag = i18n('newTag');

const withoutShowLinksCount = ['ContactCommMediumLinkTag'];

const SettingsTagsLayout = (props) => {
  const { tag, items, loadData, addTag, editTag, saveTag, resetTag, type } = props;
  const showTagForm = tag.isNew || tag.isEdit;
  const formName = `SettingsTag${(tag.data && tag.data.id) || 'New'}`;
  const grants = tag.data && tag.data.grants && tag.data.grants.map(fromGrantAdapter);

  return (
    <div className={css.root}>
      <div className={css.title}>
        {i18nTitle}
        <a className={cx(css.add, tag.isNew ? css.addActive : null)} tabIndex="0" onClick={addTag}>
          <i className="material-icons">add_box</i>
          {i18nNewTag}
        </a>
      </div>
      <div className={css.content}>
        <div className={css.tags}>
          <TagsListWithFilter
            showLinksCount={!withoutShowLinksCount.includes(type)}
            items={items}
            tag={tag}
            editTag={editTag}
          />
        </div>
        <div className={css.tag}>
          <Tip tag={tag} items={items} addTag={addTag} />
          {showTagForm && (
            <TagForm
              key={formName}
              form={formName}
              initialValues={{
                type,
                ...tag.data,
                grants,
              }}
              onSubmitted={(data) => {
                loadData().then(() => {
                  saveTag(data);
                });
              }}
              onCancel={resetTag}
            />
          )}
        </div>
      </div>
    </div>
  );
};

SettingsTagsLayout.propTypes = {
  items: PropTypes.arrayOf(PropTypes.object),
  tag: PropTypes.shape({
    data: PropTypes.instanceOf(Object),
  }),
  loadData: PropTypes.func,
  addTag: PropTypes.func,
  editTag: PropTypes.func,
  saveTag: PropTypes.func,
  resetTag: PropTypes.func,
  type: PropTypes.string,
};

SettingsTagsLayout.defaultProps = {
  items: [],
  tag: {
    data: {},
  },
  loadData: () => {},
  addTag: () => {},
  editTag: () => {},
  saveTag: () => {},
  resetTag: () => {},
  type: undefined,
};

export default SettingsTagsLayout;
