import React from 'react';
import PropTypes from 'prop-types';
import cx from 'classnames';
import TitleChain from 'modules/heading';
import createI18N from '@yandex-int/i18n';
import { MODULE_NAME as SettingsModule } from 'modules/settings/constants';
import * as keyset from './Layout.i18n';
import RulesList from '../../containers/RulesList';
import Rule from '../Rule';
import Tip from '../Tip';
import css from './styles.modules.scss';

const i18n = createI18N(keyset);
const i18nTitle = i18n('title');
const i18nNewRule = i18n('newRule');
const i18nEditing = i18n('editing');

const SettingsRulesLayout = props => {
  const { rulesData, foldersData, addRule } = props;
  const { isNew, isEdit } = rulesData;

  if (!rulesData.isLoad || !foldersData.isLoad) return null;

  return (
    <div className={css.root}>
      <TitleChain>{i18nTitle}</TitleChain>
      <div className={css.content}>
        <div className={css.rules}>
          <div className={css.header}>
            <div className={css.title}>{i18nTitle}</div>
            <a className={cx(css.add, isNew ? css.addActive : null)} onClick={addRule} tabIndex="0">
              <i className="material-icons">add_box</i>
              {i18nNewRule}
            </a>
          </div>
          <RulesList
            rules={rulesData.data && rulesData.data.rules}
            folders={foldersData.tree || {}}
            tags={foldersData.tags || []}
            isEdit={isEdit}
          />
        </div>
        <div className={css.rule}>
          <div className={css.path}>
            {SettingsModule} → {i18nTitle}
            {isNew && (
              <span>
                {' '}
                → <span className={css.isNew}>{i18nNewRule}</span>
              </span>
            )}
            {isEdit && <span> → {i18nEditing}</span>}
          </div>
          <Tip rulesData={rulesData} addRule={addRule} />
          <Rule rulesData={rulesData} foldersData={foldersData} />
        </div>
      </div>
    </div>
  );
};

SettingsRulesLayout.propTypes = {
  rulesData: PropTypes.instanceOf(Object),
  foldersData: PropTypes.instanceOf(Object),
  addRule: PropTypes.func,
};

SettingsRulesLayout.defaultProps = {
  rulesData: {},
  foldersData: {},
  addRule: () => {},
};

export default SettingsRulesLayout;
