import React from 'react';
import get from 'lodash/get';
import { Field } from 'redux-form';
import Select from 'lego/redux-form/Select';
import TextInput from 'lego/redux-form/TextInput';
import Button from '@crm/components/dist/lego2/Button';
import Icon from '@crm/components/dist/lego2/Icon';
import cx from 'classnames';
import createI18N from '@yandex-int/i18n';
import * as keyset from './RuleForm.i18n';
import { AccountSelectInput as AccountSelect } from '../FormFields/AccountSelect';
import css from './conditions.modules.scss';
import meta from './meta';

const i18n = createI18N(keyset);
const i18nIf = i18n('if');
const i18nConditionHeaderPlaceholder = i18n('conditionHeaderPlaceholder');
const i18nAddCondition = i18n('addCondition');

const SettingsRuleFormConditions = (props) => {
  const {
    fields,
    meta: { touched, error },
    conditions,
    change,
    touch,
    untouch,
    size,
  } = props;

  return (
    <div>
      <div className={css.label}>{i18nIf}</div>
      {fields.length > 1 ? (
        <div className={css.operator}>
          <div className={css.row}>
            <Field
              name="operator"
              id="operator"
              className={cx(css.field, css.control)}
              component={Select}
              options={meta.fields.operator.list.map((i) => ({
                value: i.toString(),
                content: meta.fields.operator.map[i],
              }))}
              size={size}
            />
          </div>
        </div>
      ) : null}

      {fields.map((condition, index) => {
        let displayHeaderField = false;
        let displayAccountField = false;

        if (conditions && conditions[index]) {
          if (conditions[index].conditionType === 'header') {
            displayHeaderField = true;
          }
          if (conditions[index].conditionType === 'account') {
            displayAccountField = true;
          }
        }

        return (
          <div key={index} className={css.condition}>
            <div className={cx(css.row, fields.length > 1 ? null : css.isOnly)}>
              <div className={css.control}>
                <Field
                  className={css.field}
                  componentClassName={css.field}
                  errorOffset={32}
                  name={`${condition}.conditionType`}
                  id={`${condition}.conditionType`}
                  component={Select}
                  onChange={(e, value, prevValue) => {
                    if (value === 'account') {
                      change(`${condition}.operator`, 'equal');
                      change(`${condition}.value`, '');
                    }
                    if (prevValue === 'account') {
                      change(`${condition}.setAccount`, {});
                      untouch(`${condition}.setAccount`);
                      change(`${condition}.value`, '');
                    }
                  }}
                  options={meta.fields.conditionType.list.map((i) => ({
                    value: i.toString(),
                    content: meta.fields.conditionType.map[i],
                  }))}
                  size={size}
                />
                {displayHeaderField && (
                  <Field
                    className={css.conditonHeader}
                    name={`${condition}.name`}
                    id={`${condition}.name`}
                    component={TextInput}
                    maxLength={300}
                    placeholder={i18nConditionHeaderPlaceholder}
                    size={size}
                  />
                )}
              </div>

              {!displayAccountField && (
                <Field
                  className={cx(css.field, css.control)}
                  componentClassName={cx(css.field, css.control)}
                  errorOffset={32}
                  name={`${condition}.operator`}
                  id={`${condition}.operator`}
                  component={Select}
                  options={meta.fields.conditionOperator.list.map((i) => ({
                    value: i.toString(),
                    content: meta.fields.conditionOperator.map[i],
                  }))}
                  size={size}
                />
              )}

              {displayAccountField ? (
                <Field
                  className={cx(css.value, css.account)}
                  errorOffset={32}
                  name={`${condition}.setAccount`}
                  id={`${condition}.setAccount`}
                  component={AccountSelect}
                  disabled={!actionsIsAccount}
                  viewProps={{
                    className: css.setAccount,
                    classNameButton: css.control,
                    classNameInput: css.value,
                  }}
                  defaultValue={{
                    accountId: get(conditions[index], 'value'),
                    name: get(conditions[index], 'name'),
                    login: get(conditions[index], 'login'),
                  }}
                  onChange={(e, value) => {
                    if (value.accountId) {
                      change(`${condition}.value`, value.accountId.toString());
                    }
                  }}
                  onCancel={() => {
                    touch(`${condition}.setAccount`); // чтобы показать ошибку
                  }}
                  size={size}
                />
              ) : (
                <Field
                  className={cx(css.field, css.value, displayAccountField)}
                  errorOffset={32}
                  name={`${condition}.value`}
                  id={`${condition}.value`}
                  component={TextInput}
                  maxLength={300}
                  size={size}
                />
              )}

              {fields.length > 1 ? (
                <Button
                  view="clear"
                  size={size}
                  onClick={() => fields.remove(index)}
                  icon={(iconCls) => <Icon className={iconCls} type="cross" />}
                />
              ) : null}
            </div>
          </div>
        );
      })}

      {error && <span className={css.error}>{error}</span>}

      <div>
        <Button
          view="pseudo"
          size={size}
          onClick={() => fields.push({ conditionType: 'from', operator: 'equal' })}
        >
          {i18nAddCondition}
        </Button>
        {touched && error && <span>{error}</span>}
      </div>
    </div>
  );
};

SettingsRuleFormConditions.defaultProps = {
  size: 'm',
};

export default SettingsRuleFormConditions;
