import React from 'react';
import PropTypes from 'prop-types';
import { Field, FieldArray } from 'redux-form';
import Button from '@crm/components/dist/lego2/Button';
import CheckBox from 'lego/redux-form/CheckBox';
import createI18N from '@yandex-int/i18n';
import * as commonKeyset from 'common.i18n';
import * as keyset from './RuleForm.i18n';
import RenderConditions from './Conditions';
import RenderActions from './Actions';
import css from './styles.modules.scss';
import meta from './meta';

const commonI18n = createI18N(commonKeyset);
const i18n = createI18N(keyset);
const i18nSave = commonI18n('save');
const i18nApplyToExisting = i18n('applyToExisting');
const i18nCancel = commonI18n('cancel');
const i18nRemove = commonI18n('remove');
const i18nApplyTip = i18n('applyTip');

const SettingsRuleForm = (props) => {
  const {
    handleSubmit,
    submitting,
    pristine,
    invalid,
    change,
    touch,
    untouch,
    conditions,
    data: { index, folders, tags },
    actionsIsRemove,
    actionsIsFolder,
    actionsIsTag,
    actionsIsAccount,
    onDelete,
    onReset,
    onApply,
    canImmediately,
    error,
  } = props;

  const isNew = typeof index === 'undefined';

  return (
    <form onSubmit={handleSubmit} className={css.form}>
      <div className={css.content}>
        <FieldArray
          name="conditions"
          component={RenderConditions}
          conditions={conditions}
          change={change}
          touch={touch}
          untouch={untouch}
        />

        <RenderActions
          change={change}
          data={{ folders, tags }}
          actionsIsRemove={actionsIsRemove}
          actionsIsFolder={actionsIsFolder}
          actionsIsTag={actionsIsTag}
          actionsIsAccount={actionsIsAccount}
          error={error}
        />

        <div className={css.block}>
          <Field name="skipOther" id="skipOther" className={css.skipOther} component={CheckBox}>
            {meta.fields.skipOther.caption}
          </Field>
        </div>

        <div className={css.actions}>
          <span>
            <Button
              type="submit"
              size="m"
              view="action"
              disabled={submitting || invalid || pristine}
            >
              {i18nSave}
            </Button>
            <Button
              size="m"
              view="action"
              disabled={submitting || !pristine || !canImmediately}
              onClick={onApply}
            >
              {i18nApplyToExisting}
            </Button>
            <Button size="m" view="pseudo" disabled={submitting} onClick={onReset}>
              {i18nCancel}
            </Button>
          </span>
          {!isNew ? (
            <Button size="m" view="pseudo" disabled={submitting} onClick={onDelete}>
              {i18nRemove}
            </Button>
          ) : null}
        </div>
        <div>
          <i>{i18nApplyTip}</i>
        </div>
      </div>
    </form>
  );
};

SettingsRuleForm.propTypes = {
  data: PropTypes.shape({
    index: PropTypes.number,
    folders: PropTypes.instanceOf(Object),
    tags: PropTypes.arrayOf(PropTypes.object),
  }),
  onDelete: PropTypes.func,
  onReset: PropTypes.func,
  onApply: PropTypes.func,
};

SettingsRuleForm.defaultProps = {
  data: {
    index: undefined,
    folders: {},
    tags: [],
  },
  onDelete: () => {},
  onReset: () => {},
  onApply: () => {},
};

export default SettingsRuleForm;
