import React from 'react';
import { SortableHandle } from 'react-sortable-hoc';
import createI18N from '@yandex-int/i18n';
import * as commonKeyset from 'common.i18n';
import cx from 'classnames';
import * as keyset from './RulesListItem.i18n';
import getAccountVal from '../../utils/getAccountVal';
import css from './styles.modules.scss';
import meta from '../RuleForm/meta';

const commonI18n = createI18N(commonKeyset);
const i18n = createI18N(keyset);
const i18nIf = i18n('if');
const i18nOr = i18n('or');
const i18nAnd = i18n('and');
const i18nRemove = commonI18n('remove');
const i18nMarkAsRead = i18n('markAsRead');
const i18nMarkAsImportant = i18n('markAsImportant');
const i18nSkipOtherRules = i18n('skipOtherRules');

const dragHandle = isEdit =>
  SortableHandle(() => (
    <span className={cx(css.itemDrag, isEdit ? css.itemDragDisabled : null)}>
      <i className={cx('material-icons', css.iconDrag)}>reorder</i>
    </span>
  ));

const ListItem = args => {
  const { itemIndex, item, folders, tags, isEdit, onEdit, onDelete } = args;
  const DragHandle = dragHandle(isEdit);

  return (
    <div className={cx(css.listItem, item.isEdit ? css.listItemCurrent : null)}>
      <DragHandle />
      <a
        className={css.itemContent}
        tabIndex="0"
        onClick={() => {
          onEdit(itemIndex);
        }}
      >
        {item.conditions &&
          item.conditions.map((condition, index) => (
            <span key={`condition${index}`} className={css.condition}>
              {index === 0 && <span className={css.if}>{i18nIf}</span>}
              {index > 0 && (
                <span className={css.if}>{item.operator === 'or' ? i18nOr : i18nAnd}</span>
              )}
              <span className={css.field}>
                {`«${meta.fields.conditionType.map[condition.conditionType]}»`}
              </span>
              &nbsp;
              <span className={css.behavior}>
                {meta.fields.conditionOperator.map[condition.operator]}
              </span>
              &nbsp;
              <span className={css.text}>
                {condition.conditionType === 'account'
                  ? `«${getAccountVal({
                      accountId: condition.value,
                      name: condition.name,
                      login: condition.login,
                    })}»`
                  : `«${condition.value}»`}
              </span>
            </span>
          ))}
        <span className={css.rules}>
          {item.actions && item.actions.isRemove && <span className={css.rule}>{i18nRemove}</span>}
          {item.actions && item.actions.markAsRead && (
            <span className={css.rule}>{i18nMarkAsRead}</span>
          )}
          {item.actions && item.actions.markAsImportant && (
            <span className={css.rule}>{i18nMarkAsImportant}</span>
          )}
          {item.actions && item.actions.moveToFolderId && (
            <span className={css.rule}>
              {i18n('moveToFolder', {
                folderName:
                  folders[item.actions.moveToFolderId] && folders[item.actions.moveToFolderId].name,
              })}
            </span>
          )}
          {item.actions && item.actions.markTagId && (
            <span className={css.rule}>
              {i18n('setTag', {
                tagName: (() => {
                  let result;
                  tags.some(tag => {
                    if (tag.id === item.actions.markTagId) {
                      result = tag.name;
                    }
                    return tag.id === item.actions.markTagId;
                  });
                  return result;
                })(),
              })}
            </span>
          )}{' '}
          {item.skipOther === true && <span className={css.rule}>{i18nSkipOtherRules}</span>}
          {item.actions && item.actions.setAccount && (
            <span className={css.rule}>
              {i18n('setAccount', { accountName: getAccountVal(item.actions.setAccount) })}
            </span>
          )}
        </span>
      </a>
      <span className={css.itemActions}>
        <a
          className={css.action}
          tabIndex="0"
          onClick={() => {
            onDelete(itemIndex);
          }}
        >
          <i className={cx('material-icons', css.iconDelete)}>delete</i>
        </a>
      </span>
    </div>
  );
};

export default ListItem;
