/*
eslint-disable no-underscore-dangle
*/
import { connect } from 'react-redux';
import cloneDeep from 'lodash/cloneDeep';
import { reduxForm, formValueSelector } from 'redux-form';
import createI18N from '@yandex-int/i18n';
import * as commonKeyset from 'common.i18n';
import { push } from 'modules/notificationsUI/actions';
import * as keyset from '../mailRules.i18n';
import * as actions from '../actions';
import RuleForm from '../components/RuleForm';

const i18n = createI18N(keyset);
const i18nRequired = createI18N(commonKeyset)('inputRequired');
const i18nAtLeastOneRuleError = i18n('atLeastOneRuleError');
const i18nAllConditionsRuleError = i18n('allConditionsRuleError');
const i18nOneActionError = i18n('oneActionError');
const i18nFolderError = i18n('folderError');
const i18nTagError = i18n('tagError');
const i18nAccountError = i18n('accountError');
const i18nRemoveConfirm = i18n('removeConfirm');

const validate = values => {
  const errors = {};
  const requiredFields = ['operator'];
  const requiredMessage = i18nRequired;

  requiredFields.forEach(field => {
    if (!values[field] || (values[field].hasOwnProperty('length') && !values[field].length)) {
      errors[field] = requiredMessage;
    }
  });

  if (!values.conditions || !values.conditions.length) {
    errors.conditions = { _error: i18nAtLeastOneRuleError };
  } else {
    const conditionsErrors = [];
    const requiredMessageAll = i18nAllConditionsRuleError;

    values.conditions.forEach((condition, conditionIndex) => {
      const conditionError = {};

      if (!condition || !condition.conditionType || !condition.conditionType.length) {
        conditionError.conditionType = requiredMessageAll;
      }

      if (
        condition &&
        condition.conditionType === 'header' &&
        (!condition.name || !condition.name.length)
      ) {
        conditionError.name = i18nRequired;
      }

      if (!condition || !condition.operator || !condition.operator.length) {
        conditionError.operator = requiredMessageAll;
      }

      if (!condition || !condition.value || !condition.value.length) {
        if (condition.conditionType === 'account') {
          conditionError.setAccount = requiredMessageAll;
        } else {
          conditionError.value = requiredMessageAll;
        }
      }

      if (Object.keys(conditionError).length) {
        conditionsErrors[conditionIndex] = conditionError;
      }
    });

    if (conditionsErrors.length) {
      errors.conditions = conditionsErrors;
    }
  }

  const actions = values.actions || {};

  if (
    actions.isRemove !== true &&
    actions.markAsRead !== true &&
    actions.markAsImportant !== true &&
    values.actionsIsFolder !== true &&
    values.actionsIsTag !== true &&
    values.actionsIsAccount !== true
  ) {
    errors._error = i18nOneActionError;
  }

  if (
    values.actionsIsFolder === true &&
    (!actions.moveToFolderId ||
      (Array.isArray(actions.moveToFolderId) && !actions.moveToFolderId.length))
  ) {
    errors._error = i18nFolderError;
  }

  if (
    values.actionsIsTag === true &&
    (!actions.markTagId || (Array.isArray(actions.markTagId) && !actions.markTagId.length))
  ) {
    errors._error = i18nTagError;
  }

  if (values.actionsIsAccount === true && !actions.setAccount) {
    errors._error = i18nAccountError;
  }

  return errors;
};

const mapState = () => (state, props) => {
  const formName = props.form;
  const selector = formValueSelector(formName);
  const { rule } = props.data;
  const { canImmediately } = rule;

  const {
    operator,
    conditions,
    actions,
    actionsIsFolder,
    actionsIsTag,
    actionsIsAccount,
  } = selector(
    state,
    'operator',
    'conditions',
    'actions',
    'actionsIsFolder',
    'actionsIsTag',
    'actionsIsAccount',
  );

  return {
    operator,
    conditions,
    actionsIsRemove: actions && actions.isRemove,
    actionsIsFolder,
    actionsIsTag,
    actionsIsAccount,
    canImmediately,
  };
};

const mapDispatch = (dispatch, props) => ({
  onSubmit: data => {
    const rule = cloneDeep(data);
    const { index, rules } = props.data;
    const newRules = rules.slice();

    rule.conditions.forEach(condition => {
      delete condition.setAccount;
    });

    if (rule.actions.isRemove) {
      delete rule.actions.moveToFolderId;
      delete rule.actions.markTagId;
      delete rule.actions.setAccount;
    }

    if (!rule.actionsIsFolder) {
      delete rule.actions.moveToFolderId;
    }

    if (!rule.actionsIsTag) {
      delete rule.actions.markTagId;
    }

    if (!rule.actionsIsAccount) {
      delete rule.actions.setAccount;
    }

    delete rule.isEdit;
    delete rule.actionsIsRemove;
    delete rule.actionsIsFolder;
    delete rule.actionsIsTag;
    delete rule.actionsIsAccount;

    if (index || index === 0) {
      newRules.splice(index, 1, rule);
    } else {
      newRules.push(rule);
    }

    dispatch(actions.saveRule({ rules: newRules })).then(response => {
      if (response.success === true) {
        return dispatch(actions.loadData());
      }
    });
  },
  onDelete: () => {
    const { index, rules } = props.data;
    const result = confirm(i18nRemoveConfirm);

    if ((!index && index !== 0) || !result) return;

    const newRules = rules.slice();
    newRules.splice(index, 1);

    dispatch(actions.deleteRule({ rules: newRules })).then(response => {
      if (response.success === true) {
        return dispatch(actions.loadData());
      }
    });
  },
  onReset: () => {
    dispatch(actions.resetRule());
  },
  onApply: () => {
    const { rule } = props.data;
    dispatch(actions.applyRule(rule)).then(r => {
      dispatch(push({ theme: 'success', text: r.message }));
    });
  },
});

export default connect(mapState(), mapDispatch)(reduxForm({ validate })(RuleForm));
