import React from 'react';
import cx from 'classnames';
import createI18N from '@yandex-int/i18n';
import * as keyset from './TableControl.i18n';
import GroupControl from './GroupControl';
import { Value, Scheme } from '../../types';
import css from './TableControl.scss';

const i18nNotifyMeIf = createI18N(keyset)('notifyMeIf');

interface Props {
  scheme: Scheme;
  value: Value;
  onChange: (value: Value) => void;
}

class TableControl extends React.Component<Props> {
  private handleChange = (v, e): void => {
    const { value, onChange } = this.props;
    const { target } = e;
    const { event, role, method, controlType } = target.dataset;

    const newValue = { ...value };
    newValue[event] = { ...newValue[event] };
    newValue[event][role] = newValue[event][role] || [];

    if (v) {
      newValue[event][role].push(method);
    } else {
      const filteredArray = newValue[event][role].filter((m) => m !== method);

      if (controlType === 'radio' && !filteredArray.length) {
        return;
      }

      newValue[event][role] = filteredArray;
    }

    if (onChange) {
      onChange(newValue);
    }
  };

  public render() {
    const { scheme, value } = this.props;

    if (!scheme) {
      return null;
    }

    return (
      <table className={css.b}>
        <tbody>
          <tr className={css.b__row}>
            <td className={css.b__cell} />
            <td className={cx(css.b__cell, css.b__cell_title)} colSpan={3}>
              {i18nNotifyMeIf}
            </td>
          </tr>
          <tr className={cx(css.b__row, css.b__row_header)}>
            <td className={cx(css.b__cell, css.b__cell_header)} />
            {scheme.roles.map((role) => (
              <td key={role.code} className={cx(css.b__cell, css.b__cell_header)}>
                {role.title}
              </td>
            ))}
          </tr>
          {scheme.events.map((event) => (
            <tr className={cx(css.b__row, css.b__row_control)} key={event.code}>
              <td className={cx(css.b__cell, css.b__cell_event)}>{event.title}</td>
              {scheme.roles.map((role) => (
                <td key={role.code} className={cx(css.b__cell, css.b__cell_control)}>
                  <GroupControl
                    disabled={Boolean(role.disabledEventsCodes?.includes(event.code))}
                    onChange={this.handleChange}
                    scheme={scheme}
                    event={event.code}
                    controlType={event.controlType}
                    role={role.code}
                    value={value}
                  />
                </td>
              ))}
            </tr>
          ))}
        </tbody>
      </table>
    );
  }
}

export default TableControl;
