import React, { Component } from 'react';
import PropTypes from 'prop-types';
import { connect } from 'react-redux';
import get from 'lodash/get';
import routeController from 'modules/settings/utils/routeController';
import { fromGrantAdapter } from 'lego/components/SuggestStaff';
import { NAME, ROUTE } from '../constants';
import * as actions from '../actions';
import TemplateForm from './TemplateForm';

const mapState = state => ({
  template: get(state, `${NAME}.template`),
});

const mapDispatch = dispatch => ({
  addTemplate: () => dispatch(actions.addTemplate()),
  editTemplate: id => dispatch(actions.loadTemplate(id)),
  resetTemplate: () => dispatch(actions.resetTemplate()),
});

class SettingsTemplate extends Component {
  constructor(props) {
    super(props);

    this.routeController = routeController({
      add: props.addTemplate,
      edit: props.editTemplate,
      reset: () => {
        const { resetTemplate, router } = props;

        resetTemplate();
        router.push(ROUTE);
      },
    });
  }

  componentDidMount() {
    const { id } = this.props.routeParams;

    this.routeController({
      isNew: id === 'new',
      id,
    });
  }

  componentWillReceiveProps(newProps) {
    if (newProps.routeParams.id !== this.props.routeParams.id) {
      const { id } = newProps.routeParams;

      this.routeController({
        isNew: id === 'new',
        id,
      });
    }
  }

  componentWillUnmount() {
    this.props.resetTemplate();
  }

  render() {
    const { template, router } = this.props;

    if ((!template.isNew && !template.isEdit) || (!template.isNew && !template.isLoad)) return null;

    const { data } = template;
    const templateId = data && data.templateId;
    const formName = `SettingsTemplate${templateId || 'New'}`;
    const grants = data && data.grants && data.grants.map(fromGrantAdapter);

    return (
      <TemplateForm
        key={formName}
        form={formName}
        initialValues={{
          ...data,
          grants,
        }}
        templateId={templateId}
        router={router}
      />
    );
  }
}

SettingsTemplate.propTypes = {
  routeParams: PropTypes.instanceOf(Object).isRequired,
  template: PropTypes.shape({
    data: PropTypes.instanceOf(Object),
  }),
};

SettingsTemplate.defaultProps = {
  template: {},
};

export default connect(mapState, mapDispatch)(SettingsTemplate);
