import * as React from 'react';
import TitleChain from 'modules/heading';
import createI18N from '@yandex-int/i18n';
import Path from 'modules/settings/modules/supports/components/Path';
import Button from '@crm/components/dist/lego2/Button';
import { Form } from 'react-final-form';
import arrayMutators from 'final-form-arrays';
import { FieldArray } from 'react-final-form-arrays';
import { get, jsonApiCall } from 'api/common';
import * as commonKeyset from 'common.i18n';
import { MODULE_NAME as SettingsModule } from 'modules/settings/constants';
import Condition from '../Condition';
import { createConditionAdapter, ConditionAdapter } from './adapters';
import css from './styles.module.css';
import * as keyset from '../../upsale.i18n';

const i18n = createI18N(keyset);
const commonI18n = createI18N(commonKeyset);
const i18nUpsale = i18n('upsale');
const i18nRole = i18n('role');
const i18nModule = i18n('module');
const i18nTier = i18n('tier');
const i18nAddCondition = i18n('addCondition');
const i18nSave = commonI18n('save');

interface State {
  initialValues?: InitialValues;
}

const mutators = { ...arrayMutators };

interface InitialValues {
  conditions: any[];
}

interface Dictionary {
  items: {
    id: number;
    name: string;
  }[];
}

interface TierData {
  includeNoValue?: boolean;
}

const getDictionary = <T extends {}>(url: string, data?: T): Promise<Dictionary> =>
  get({
    url,
    data,
  });

class Layout extends React.Component<{}, State> {
  private conditionAdapter: ConditionAdapter;

  private userRoles: Dictionary;

  private modules: Dictionary;

  private tiers: Dictionary;

  public constructor(props) {
    super(props);
    this.state = {};
  }

  public async componentDidMount() {
    const [initialValues, userRoles, tiers, modules]: [
      InitialValues,
      Dictionary,
      Dictionary,
      Dictionary,
    ] = await Promise.all([
      get({
        url: '/view/usersettings/upsaleoff',
      }),

      getDictionary('/view/dicts/userRole'),

      getDictionary<TierData>('/view/dicts/tier', { includeNoValue: true }),

      getDictionary('/view/dicts/upsaleModule'),
    ]);

    this.userRoles = userRoles;
    this.modules = modules;
    this.tiers = tiers;
    this.conditionAdapter = createConditionAdapter({
      userRoles,
      modules,
      tiers,
    });
    this.setState({
      initialValues: {
        conditions: (initialValues.conditions || []).map(this.conditionAdapter),
      },
    });
  }

  private handleSubmit = (values) => {
    const conditions = values.conditions.map(this.conditionAdapter);

    jsonApiCall({
      url: '/view/usersettings/upsaleoff/save',
      data: {
        conditions,
      },
    });
  };

  public render(): React.ReactNode {
    return (
      <div className={css.root}>
        <TitleChain>{i18nUpsale}</TitleChain>
        <Path>
          {SettingsModule} → {i18nUpsale}
        </Path>

        <Form
          onSubmit={this.handleSubmit}
          initialValues={this.state.initialValues}
          mutators={mutators}
          render={(props) => {
            return (
              <form onSubmit={props.handleSubmit}>
                <div className={css.row}>
                  <div className={css.column}>
                    <h4 className={css.column__title}>{i18nRole}</h4>
                  </div>
                  <div className={css.column}>
                    <h4 className={css.column__title}>{i18nModule}</h4>
                  </div>
                  <div className={css.column}>
                    <h4 className={css.column__title}>{i18nTier}</h4>
                  </div>
                </div>

                <FieldArray name="conditions">
                  {({ fields }) =>
                    fields.map((name, index) => (
                      <div className={css.row}>
                        <Condition
                          name={name}
                          userRoles={this.userRoles}
                          modules={this.modules}
                          tiers={this.tiers}
                          onRemoveClick={() => fields.remove(index)}
                        />
                      </div>
                    ))
                  }
                </FieldArray>

                <div className={css.row}>
                  <Button
                    className={css.addButton}
                    onClick={() => props.form.mutators.push('conditions', {})}
                    view="default"
                    disabled={props.submitting}
                  >
                    {i18nAddCondition}
                  </Button>

                  <Button
                    type="submit"
                    view="action"
                    className={css.saveButton}
                    disabled={props.submitting}
                  >
                    {i18nSave}
                  </Button>
                </div>
              </form>
            );
          }}
        />
      </div>
    );
  }
}

export default Layout;
