interface Dictionary {
  items: {
    id: number;
    name: string;
  }[];
}

interface BackendCondition {
  userRoles: number[];
  modules: number[];
  tiers: number[];
}

interface SaveBackendData {
  conditions: BackendCondition[];
}

interface FrontSuggestItem {
  id: number;
  name: string;
}

interface FrontCondition {
  userRoles: FrontSuggestItem[];
  modules: FrontSuggestItem[];
  tiers: FrontSuggestItem[];
}

const itemToId = (item: FrontSuggestItem): number => Number(item.id);
const idToItem = (array: FrontSuggestItem[], id: number): FrontSuggestItem => {
  return array.find(item => item.id === id) as FrontSuggestItem;
};

const isFrontCondition = (
  condition: BackendCondition | FrontCondition,
): condition is FrontCondition => {
  const { userRoles, modules, tiers } = condition;

  if (userRoles && userRoles.length && typeof userRoles[0] === 'number') {
    return false;
  }

  if (modules && modules.length && typeof modules[0] === 'number') {
    return false;
  }

  if (tiers && tiers.length && typeof tiers[0] === 'number') {
    return false;
  }

  return true;
};

type CreateConditionAdapter = (dictionaries: {
  userRoles: Dictionary;
  modules: Dictionary;
  tiers: Dictionary;
}) => ConditionAdapter;

export interface ConditionAdapter {
  (condition: BackendCondition): FrontCondition;
  (condition: FrontCondition): BackendCondition;
}

export const createConditionAdapter: CreateConditionAdapter = dictionaries => {
  const conditionAdapter = condition => {
    if (isFrontCondition(condition)) {
      const userRoles = (condition.userRoles || []).map(itemToId);
      const tiers = (condition.tiers || []).map(itemToId);
      const modules = (condition.modules || []).map(itemToId);

      return {
        userRoles,
        modules,
        tiers,
      };
    }

    const userRoles = (condition.userRoles || []).map(id =>
      idToItem(dictionaries.userRoles.items, id),
    );
    const modules = (condition.modules || []).map(id => idToItem(dictionaries.modules.items, id));
    const tiers = (condition.tiers || []).map(id => idToItem(dictionaries.tiers.items, id));

    const frontCondition: FrontCondition = {
      userRoles,
      modules,
      tiers,
    };
    return frontCondition;
  };

  return conditionAdapter as ConditionAdapter;
};
