import React, { Component } from 'react';
import { connect } from 'react-redux';
import { Field, reduxForm, formValueSelector, focus } from 'redux-form';
import Button from '@crm/components/dist/lego2/Button';
import debounce from 'lodash/debounce';
import cx from 'classnames';
import { isTrackerUrl, getIssueKeyFromTrackerUrl } from 'utils/trackerUrl';
import { baseCallApi } from 'api/common';
import { Title } from '@crm/components/dist/Title';
import Label from 'lego/components/Label';
import css from 'requestForms/BaseForm/style.scss';
import { push } from 'modules/notificationsUI/actions';
import TextInput from 'lego/redux-form/TextInput';
import Toolbar from 'components/Toolbar';
import formCss from 'forms/Base/styles.modules.scss';
import errorMessages from 'requestForms/BaseForm/errors';
import ticketCss from './ticket.modules.scss';

const FORM_NAME = 'ST_CONNECT';

// state of async validation
const ASYNC_VALIDATION = {
  INIT: 0,
  SUCCESS: 1,
  EXIST: 2,
  ERROR: 3,
};

// initial state of form
const INIT = {
  asyncValidation: ASYNC_VALIDATION.INIT,
  title: undefined,
};

const validate = (values) => {
  const errors = {};
  if (!values.key) {
    errors.key = errorMessages.required;
  }
  return errors;
};

const mapState = (state) => {
  const formValues = formValueSelector(FORM_NAME);
  return {
    form: FORM_NAME,
    stKey: formValues(state, 'key'),
  };
};

const mapDispatch = (dispatch, props) => ({
  onSubmit: (values) => {
    const issueKey = isTrackerUrl(values.key) ? getIssueKeyFromTrackerUrl(values.key) : values.key;

    const data = {
      ...values,
      key: issueKey,
      requestId: props.requestId,
      ticketId: props.ticketId,
      issueId: props.issueId,
    };

    if (props.onSubmit) {
      return props.onSubmit(data);
    }

    return baseCallApi({
      url: '/actions/startrek/issue/link',
      data,
      type: 'post',
    });
  },
  onSubmitSuccess: (data) => {
    if (!props.disabledMessages) {
      if (data.linkExists) {
        dispatch(
          push({
            theme: 'warn',
            text: 'ST тикет уже связан',
          }),
        );
      } else {
        dispatch(
          push({
            theme: 'success',
            text: `ST тикет ${data.issue.key} связан`,
          }),
        );
      }
    }

    if (props.onSubmitSuccess) {
      props.onSubmitSuccess();
    }
  },
});

@connect(mapState, mapDispatch)
@reduxForm({
  validate,
})
export default class ConnectTicketForm extends Component {
  constructor(props) {
    super(props);

    this.state = INIT;

    this.asyncValidate = debounce(this.asyncValidate, 300);
  }

  componentWillMount() {
    this.props.dispatch(focus(this.props.form, 'key'));
  }

  componentWillReceiveProps(nextProps) {
    if (nextProps.stKey !== this.props.stKey) {
      this.setState(INIT);
    }

    if (nextProps.stKey && nextProps.stKey !== this.props.stKey) {
      this.asyncValidate(nextProps.stKey);
    }
  }

  asyncValidate = (keyOrUrl) => {
    let key = keyOrUrl;
    const keyIsTrackerUrl = isTrackerUrl(keyOrUrl);
    if (keyIsTrackerUrl) {
      key = getIssueKeyFromTrackerUrl(keyOrUrl);
    }

    return baseCallApi({
      url: '/view/startrek/issue/find',
      data: {
        key,
        requestId: this.props.requestId,
        ticketId: this.props.ticketId,
        issueId: this.props.issueId,
      },
      type: 'post',
    }).then((data) => {
      if (key === this.props.stKey || keyIsTrackerUrl) {
        if (data.alreadyLinked) {
          this.setState({
            asyncValidation: ASYNC_VALIDATION.EXIST,
          });
        } else if (data.found) {
          this.setState({
            asyncValidation: ASYNC_VALIDATION.SUCCESS,
            title: data.issue.title,
          });
        } else {
          this.setState({
            asyncValidation: ASYNC_VALIDATION.ERROR,
          });
        }
      }
    });
  };

  handleOnChange = () => {
    this.props.asyncValidate();
  };

  render() {
    const { handleSubmit, pristine, submitting, valid } = this.props;
    const isValid =
      !pristine && !submitting && valid && this.state.asyncValidation === ASYNC_VALIDATION.SUCCESS;

    const renderMessage = () => {
      switch (this.state.asyncValidation) {
        case ASYNC_VALIDATION.SUCCESS:
          return (
            <div className={ticketCss.root}>
              <div className={ticketCss.found}>Тикет найден</div>
              <div>
                <span>Название: </span>
                {this.state.title}
              </div>
            </div>
          );
        case ASYNC_VALIDATION.ERROR:
          return (
            <div className={ticketCss.root}>
              <span className={ticketCss.notfound}>Тикет не найден</span>
            </div>
          );
        case ASYNC_VALIDATION.EXIST:
          return (
            <div className={ticketCss.root}>
              <span className={ticketCss.exist}>Тикет уже связан</span>
            </div>
          );
        default:
          return null;
      }
    };

    return (
      <form className={cx(css.form, 'isRegular', formCss.form)} onSubmit={handleSubmit}>
        <Title>Связать с тикетом ST</Title>
        <div className={formCss.hint}>
          {' '}
          Поля обозначенные <span className={css.requiredStar} /> – обязательные
        </div>
        <div className={formCss.row}>
          <div className={formCss.rowLabel}>
            <Label required className={cx(css.formLabel)}>
              ST тикет
            </Label>
          </div>
          <div className={css.rowField}>
            <Field className={css.formField} name="key" component={TextInput} size="m" />
          </div>
        </div>
        {renderMessage()}
        <div className={formCss.footer}>
          <Toolbar reverse>
            <Button
              type="submit"
              view="action"
              disabled={!isValid}
              onClick={this.handleOK}
              size="m"
            >
              Связать
            </Button>
            <Button onClick={this.props.onCancel} size="m">
              Отменить
            </Button>
          </Toolbar>
        </div>
      </form>
    );
  }
}
