import React, { Component, FC } from 'react';
import { Field, AnyObject } from 'react-final-form';
import { Form, FieldRow, Condition } from 'components/FinalForm';
import AttachFiles from 'lego/final-form/AttachFiles';
import formCss from 'forms/Base/styles.modules.scss';
import errorMessages from 'requestForms/BaseForm/errors';
import enchanceApi from './enhanceApi';
import { InnerNewTicketFormProps } from './NewTicketForm.types';

const validate = (values, props) => {
  const errors: AnyObject = {};

  const requiredFields = ['queue', 'title', 'type'];
  if (props.requiredFields.componentIds && !(values.componentIds && values.componentIds.length)) {
    errors.componentIds = errorMessages.required;
  }

  requiredFields.forEach((field) => {
    if (values[field] === undefined) {
      errors[field] = errorMessages.required;
    }
  });

  return errors;
};

const exists = (value: unknown): boolean => Boolean(value);

class NewTicketForm extends Component<InnerNewTicketFormProps> {
  static defaultProps = {
    size: 'm',
  };

  handleSubmit = ({ assignTo, ...otherValues }) =>
    this.props.onSubmit({ ...otherValues, assignTo: assignTo && assignTo.id });

  validate = (values) => validate(values, this.props);

  suggestItemComponent: FC<{
    item: string;
  }> = (props) => <>{props.item}</>;

  suggestGetKey = (item: string) => item;

  render() {
    const { size, initialValues } = this.props;

    const queueProvider = (rawText) => {
      const text = rawText.trim().toLowerCase();
      return this.props.queueList.result.filter((rawName) => {
        const name = rawName.toLowerCase();
        return name.indexOf(text) !== -1;
      });
    };

    return (
      <Form
        title="Создать новый тикет ST"
        onSubmitSuccess={this.props.onSubmitSuccess}
        initialValues={initialValues}
        onCancel={this.props.onCancel}
        cancelButtonText="Отменить"
        successButtonText="Создать"
        onSubmit={this.handleSubmit}
        validate={this.validate}
      >
        <FieldRow
          name="queue"
          component="Suggest"
          label="Очередь"
          inputProps={{
            component: this.suggestItemComponent,
            tagComponent: this.suggestItemComponent,
            provider: queueProvider,
            showOnFocus: true,
            isSingleValue: true,
            getKey: this.suggestGetKey,
          }}
          hasEmptyValue={false}
          required
          size={size}
        />

        <Condition when="queue" is={exists}>
          {(queue) => (
            <>
              <FieldRow
                name="type"
                component="SelectWithLoad"
                label="Тип задачи"
                hasEmptyValue={false}
                required
                size={size}
                endpoint={`/view/startrek/component/list?queue=${queue}`}
                useCache={false}
                loadedListKey="issyetypes"
              />
              <FieldRow
                name="componentIds"
                component="SelectWithLoad"
                label="Компоненты"
                maxHeight={400}
                hasEmptyValue={false}
                multiple
                size={size}
                endpoint={`/view/startrek/component/list?queue=${queue}`}
                useCache={false}
              />
            </>
          )}
        </Condition>

        <FieldRow
          name="title"
          component="TextInput"
          label="Название задачи"
          required
          hasClear
          size={size}
        />

        <FieldRow name="text" component="TextArea" label="Описание" rows="10" size={size} />

        <FieldRow
          name="assignTo"
          component="SuggestStaff"
          label="Назначить"
          isSingleValue
          size={size}
        />

        <Field
          name="files"
          className={formCss.attachFiles}
          objectName="container"
          component={AttachFiles}
        />
      </Form>
    );
  }
}

export default enchanceApi(NewTicketForm);
