import React, { Component } from 'react';
import PropTypes from 'prop-types';
import Popup from '@crm/components/dist/lego2/Popup';
import Button from '@crm/components/dist/lego2/Button';
import cx from 'classnames';
import createI18N from '@yandex-int/i18n';
import Icon from '@crm/components/dist/lego2/Icon';
import * as keyset from './SetTag.i18n';
import SetTagList from '../SetTagList';
import connectNewTagModal from '../../containers/connectNewTagModal';
import NewTagModal from '../NewTagModal';
import css from './styles.modules.scss';
import { getTags } from '../../actions';

const i18n = createI18N(keyset);
const i18nNewTag = i18n('newTag');
const i18nAddTag = i18n('addTag');

const RenderModal = connectNewTagModal(NewTagModal);

const iconProvider = (iconCls) => <Icon className={iconCls} svg="addMg" svgSize="ms" />;

class SetTag extends Component {
  constructor(props) {
    super(props);
    this.state = {
      popup: false,
      modal: false,
      tags: [],
    };
    this.node = React.createRef();
  }

  setTag = (item) => {
    this.props.addTag(item);
    this.hidePopup();
  };

  showPopup = () => {
    const { type } = this.props;

    let promise;
    if (type) {
      promise = getTags({ type });
    } else {
      promise = this.props.loadTags();
    }

    promise.then((data) => {
      this.setState({
        tags: data && data.items,
        popup: true,
        modal: false,
      });
    });
  };

  hidePopup = () => {
    this.setState({ popup: false });
  };

  showModal = () => {
    this.setState({ modal: true, popup: false });
  };

  hideModal = () => {
    this.setState({ modal: false });
  };

  renderPopup() {
    return (
      <div className={css.popupContent}>
        <SetTagList className={css.list} tags={this.state.tags} onClick={this.setTag} />
        {this.props.canCreate && (
          <div className={css.addNew}>
            <Button size="xs" onClick={this.showModal} view="pseudo">
              {i18nNewTag}
            </Button>
          </div>
        )}
      </div>
    );
  }

  render() {
    const { type, formFields } = this.props;

    let initialValues;
    if (type) {
      initialValues = { type };
    }

    return (
      <span className={cx(this.props.className, css.root)}>
        <span className={css.add}>
          <Button
            size="xs"
            view="clear"
            innerRef={this.node}
            title={i18nAddTag}
            onClick={this.showPopup}
            icon={iconProvider}
          />
        </span>
        <Popup
          visible={this.state.popup}
          onOutsideClick={this.hidePopup}
          target="anchor"
          anchor={this.node}
          autoZIndex
          minZIndex={2003}
          directions="bottom"
        >
          {this.renderPopup()}
        </Popup>
        <RenderModal
          form="AddNewTag"
          fields={formFields}
          show={this.state.modal}
          onHide={this.hideModal}
          addTag={this.props.addTag}
          initialValues={initialValues}
        />
      </span>
    );
  }
}

// todo: add support placement

SetTag.propTypes = {
  className: PropTypes.string,
  type: PropTypes.string,
  loadTags: PropTypes.func,
  addTag: PropTypes.func,
  canCreate: PropTypes.bool,
};

SetTag.defaultProps = {
  className: undefined,
  type: undefined,
  loadTags: () => {},
  addTag: () => {},
  canCreate: true,
};

export default SetTag;
