import React from 'react';
import PropTypes from 'prop-types';
import cx from 'classnames';
import Menu from '@crm/components/dist/lego2/Menu';
import createI18N from '@yandex-int/i18n';
import Textinput from '@crm/components/dist/lego2/Textinput';
import * as keyset from './SetTagList.i18n';
import css from './styles.modules.scss';

const i18n = createI18N(keyset);
const i18nSharedTags = i18n('sharedTags');
const i18nPersonalTags = i18n('personalTags');
const i18nNoTags = i18n('noTags');
const i18nSearch = i18n('search');
const i18nSearchNoTags = i18n('searchNoTags');

const TRANSLATE = {
  Shared: i18nSharedTags,
  Personal: i18nPersonalTags,
  Default: undefined,
};

const SetTagsList = ({ className, tags, onClick, onSearch, search, isEmpty, isEmptySearch }) => {
  if (isEmpty) return <div className={cx(css.empty, className)}>{i18nNoTags}</div>;

  // собираем группы меток из списка для вывода
  const groups = tags.reduce(
    // eslint-disable-next-line no-shadow
    (groups, item) => {
      const type = TRANSLATE[item.type] ? item.type : 'Default';
      if (groups.map[type]) {
        groups.map[type].push(item);
      } else {
        groups.list.push(type);
        groups.map[type] = [item]; // eslint-disable-line no-param-reassign
      }

      return groups;
    },
    { list: [], map: {} },
  );

  return (
    <div>
      <div className={css.b__searchWrap}>
        <Textinput
          autoFocus
          size="xs"
          width="auto"
          placeholder={i18nSearch}
          value={search}
          onChange={onSearch}
        />
      </div>
      <div className={cx(css.b, className)}>
        {isEmptySearch ? (
          <div className={cx(css.empty, css.empty_search)}>{i18nSearchNoTags}</div>
        ) : (
          <Menu
            size="xs"
            width="max"
            onChange={(event) => onClick(event.target.value)}
            items={groups.list.map((groupId) => ({
              title: TRANSLATE[groupId],
              items: groups.map[groupId].map((item) => ({
                value: item,
                content: (
                  <span>
                    <span
                      className={css.colorBox}
                      style={{ background: `#${item.color || 'transparent'}` }}
                    />
                    &nbsp;
                    <span dangerouslySetInnerHTML={{ __html: item.nameWithSearch || item.name }} />
                  </span>
                ),
              })),
            }))}
          />
        )}
      </div>
    </div>
  );
};

SetTagsList.propTypes = {
  className: PropTypes.string,
  search: PropTypes.string.isRequired,
  onClick: PropTypes.func.isRequired,
  onSearch: PropTypes.func.isRequired,
  isEmpty: PropTypes.bool,
  isEmptySearch: PropTypes.bool,
  tags: PropTypes.instanceOf(Object),
};

SetTagsList.defaultProps = {
  className: undefined,
  tags: undefined,
  isEmpty: false,
  isEmptySearch: false,
};

export default SetTagsList;
