import * as React from 'react';
import escapeRegExp from 'lodash/escapeRegExp';
import SetTagList from './SetTagList';

interface Tag {
  id: number;
  name: string;
}

interface TagWithSearch extends Tag {
  nameWithSearch: string;
}

interface Props {
  tags: Tag[];
}

interface State {
  tags: Tag[];
  search: string;
}

class SetTagListController extends React.Component<Props, State> {
  public static defaultProps = {
    tags: [],
  };

  public constructor(props) {
    super(props);

    this.state = {
      search: '',
      tags: props.tags,
    };
  }

  private handleSearch = (value: string) => {
    this.setState({ search: value });
    this.filter(value);
  };

  private filter = (str: string) => {
    const { tags } = this.props;

    if (!str) {
      this.setState({ tags });
    } else {
      const searchPattern = new RegExp(`(${escapeRegExp(str)})`, 'gim');
      const filteredTags: TagWithSearch[] = [];

      tags.forEach(tag => {
        const { name } = tag;
        if (searchPattern.test(name)) {
          filteredTags.push({
            ...tag,
            nameWithSearch: name.replace(
              searchPattern,
              match => `<span style="background-color: rgba(0, 173, 255, 0.3);">${match}</span>`,
            ),
          });
        }
      });

      this.setState({ tags: filteredTags });
    }
  };

  public render() {
    const { tags, ...passPropsThrough } = this.props;
    const { tags: filteredTags } = this.state;

    const isEmpty = !(tags && tags.length);
    const isEmptySearch = !(filteredTags && filteredTags.length);

    return React.createElement(SetTagList, {
      ...passPropsThrough,
      ...this.state,
      isEmpty,
      isEmptySearch,
      onSearch: this.handleSearch,
    });
  }
}

export default SetTagListController;
