import React, { Component } from 'react';
import PropTypes from 'prop-types';
import { connect } from 'react-redux';
import isEqual from 'lodash/isEqual';
import get from 'lodash/get';
import cx from 'classnames';
import { NAME } from '../../constants';
import reducer from '../../reducer';
import * as actions from '../../actions';
import SetTag from '../../components/SetTag/index';
import css from './styles.modules.scss';

window.reduxStore.injectReducer(NAME, reducer);

const withManage = WrappedComponent => {
  class TagsWithManage extends Component {
    componentWillMount() {
      this.init();
    }

    componentWillReceiveProps(newProps) {
      if (this.props.enableReinitialize) {
        let tagsChanged = false;

        if (
          (this.props.tags && !newProps.tags) ||
          (!this.props.tags && newProps.tags) ||
          this.props.tags.length !== newProps.tags.length
        ) {
          tagsChanged = true;
        } else if (
          this.props.tags &&
          newProps.tags &&
          this.props.tags.length === newProps.tags.length
        ) {
          tagsChanged = !isEqual(this.props.tags, newProps.tags);
        }

        if (tagsChanged) {
          this.props.update(newProps.tags);
        }
      }
    }

    componentDidUpdate(prevProps) {
      this.init(prevProps);
    }

    componentWillUnmount() {
      if (this.isInit() && this.props.destroyOnUnmount) {
        this.props.destroy();
      }
    }

    getTags() {
      return this.isInit() ? this.props.tagsData.data : this.props.tags;
    }

    isRender() {
      return this.props.asyncInit || this.isInit();
    }

    isInit() {
      return !!this.props.tagsData;
    }

    init(prevProps = {}) {
      const { asyncInit, needInit, tags, init } = this.props;
      if (!this.isInit() && (!asyncInit || (asyncInit && needInit && !prevProps.needInit))) {
        init(tags);
      }
    }

    add = tag => {
      const { addTag } = this.props;

      if (typeof addTag === 'function') {
        addTag(tag.id).then(response => {
          if (response.success) {
            this.props.add(response.tag);
          }

          return response;
        });
      }
    };

    delete = id => {
      const { removeTag } = this.props;

      if (typeof removeTag === 'function') {
        removeTag(id).then(response => {
          if (response.success) {
            this.props.delete(id);
          }

          return response;
        });
      }
    };

    render() {
      const {
        className,
        addClassName,
        tagsClassName,
        name,
        getTags,
        addTag,
        removeTag,
        canEdit,
        canCreate,
        tags,
        tagsData,
        placement,
        ...passThroughProps
      } = this.props;

      if (!this.isRender()) return null;

      return (
        <div className={cx(css.root, className)}>
          {canEdit && getTags && typeof getTags === 'function' && (
            <SetTag
              className={addClassName}
              loadTags={getTags}
              addTag={this.add}
              placement={placement}
              canCreate={canCreate}
            />
          )}
          <WrappedComponent
            className={tagsClassName}
            canEdit={canEdit}
            onDelete={canEdit ? this.delete : undefined}
            tags={this.getTags()}
            {...passThroughProps}
          />
        </div>
      );
    }
  }

  TagsWithManage.propTypes = {
    className: PropTypes.string,
    addClassName: PropTypes.string,
    tagsClassName: PropTypes.string,
    tags: PropTypes.arrayOf(PropTypes.object),
    name: PropTypes.string.isRequired,
    getTags: PropTypes.func,
    addTag: PropTypes.func,
    removeTag: PropTypes.func,
    canEdit: PropTypes.bool,
    canCreate: PropTypes.bool,
    destroyOnUnmount: PropTypes.bool,
    enableReinitialize: PropTypes.bool,
    tagsData: PropTypes.shape({
      data: PropTypes.arrayOf(PropTypes.object),
    }),
    placement: PropTypes.oneOf(['top', 'right', 'bottom', 'left']),
    init: PropTypes.func,
    update: PropTypes.func,
    destroy: PropTypes.func,
    add: PropTypes.func,
    delete: PropTypes.func,
    asyncInit: PropTypes.bool,
    needInit: PropTypes.bool,
  };

  TagsWithManage.defaultProps = {
    className: undefined,
    addClassName: undefined,
    tagsClassName: undefined,
    asyncInit: undefined,
    needInit: undefined,
    tags: [],
    getTags: undefined,
    addTag: undefined,
    removeTag: undefined,
    canEdit: true,
    canCreate: true,
    destroyOnUnmount: undefined,
    enableReinitialize: undefined,
    tagsData: undefined,
    placement: 'bottom',
    init: () => {},
    update: () => {},
    destroy: () => {},
    add: () => {},
    delete: () => {},
  };

  const mapState = (state, props) => ({
    tagsData: get(state, `${NAME}.${props.name}`),
  });

  const mapDispatch = (dispatch, props) => {
    if (!props.name) return {};

    const bindName = actionCreator => actionCreator.bind(null, props.name);

    return {
      init: data => dispatch(bindName(actions.initTags)(data)),
      update: data => dispatch(bindName(actions.updateTags)(data)),
      destroy: () => dispatch(bindName(actions.destroyTags)()),
      add: tag => dispatch(bindName(actions.addTag)(tag)),
      delete: tagId => dispatch(bindName(actions.deleteTag)(tagId)),
    };
  };

  return connect(mapState, mapDispatch)(TagsWithManage);
};

export default withManage;
