import React from 'react';
import cx from 'classnames';
import Button from '@crm/components/dist/lego2/Button';
import Icon from '@crm/components/dist/lego2/Icon';
import { ImperativePopup } from '@crm/components/dist/lego2/Popup';
import createI18N from '@yandex-int/i18n';
import { Direction } from '@crm/components/dist/lego2/withOldDirections';
import { Template } from 'types/TemplateProvider';
import * as keyset from './Templates.i18n';
import TemplatesList from './TemplatesList';
import css from './styles.module.css';
import { TemplatesProps } from './Template.types';

const i18nTemplates = createI18N(keyset)('templates');

export * from './Template.types';

const TEMPLATES_DIRECTION_TO_POPUP_DIRECTIONS: Record<'top' | 'bottom', Direction[]> = {
  top: ['top-center'],
  bottom: ['bottom-right'],
};

export default class Templates extends React.Component<TemplatesProps> {
  public static defaultProps = {
    size: 's',
    direction: 'bottom',
  };

  public componentDidMount() {
    this.forceUpdate();
  }

  // eslint-disable-next-line @typescript-eslint/no-explicit-any
  private popupBag: any;

  private renderIcon = (iconCls) => (
    <Icon className={iconCls} type="arrow" direction={this.popupBag.visible ? 'top' : 'bottom'} />
  );

  private toggle = (popupBag) => {
    this.popupBag = popupBag;
    const { size, disabled } = this.props;

    return (
      <Button size={size} disabled={disabled} iconRight={this.renderIcon}>
        {i18nTemplates}
      </Button>
    );
  };

  private handleTemplateClick = (template: Template) => {
    this.props.onTemplateClick(template);

    this.popupBag.hide();
  };

  private handleKeyUp = (e) => {
    switch (e.key) {
      case 'Escape':
        e.nativeEvent.stopImmediatePropagation();
        e.preventDefault();
        this.popupBag.hide();
        break;
      default:
    }
  };

  private getIsPlain = () => this.props.isPlain;

  private content = (popupBag) => {
    this.popupBag = popupBag;

    return (
      <TemplatesList
        isPlain={this.getIsPlain}
        templates={this.props.templates}
        onTemplateClick={this.handleTemplateClick}
        onKeyUp={this.handleKeyUp}
      />
    );
  };

  public render() {
    return (
      <ImperativePopup
        className={cx(css.popup, css[`popup_${this.props.direction}`])}
        toggle={this.toggle}
        content={this.content}
        scope={this.props.container}
        directions={TEMPLATES_DIRECTION_TO_POPUP_DIRECTIONS[this.props.direction]}
        zIndex={100}
      />
    );
  }
}
