import React from 'react';
import cx from 'classnames';
import PropTypes from 'prop-types';
import Button from '@crm/components/dist/lego2/Button';
import Icon from '@crm/components/dist/lego2/Icon';
import { ImperativePopup } from '@crm/components/dist/lego2/Popup';
import createI18N from '@yandex-int/i18n';
import * as keyset from './Templates.i18n';
import TemplatesList from './TemplatesList';
import css from './styles.module.css';

const i18nTemplates = createI18N(keyset)('templates');

export default class Templates extends React.Component {
  componentDidMount() {
    this.forceUpdate();
  }

  toggle = (popupBag) => {
    const { ckeditor, size } = this.props;

    return (
      <Button
        size={size}
        disabled={!ckeditor}
        iconRight={(iconCls) => (
          <Icon className={iconCls} type="arrow" direction={popupBag.visible ? 'top' : 'bottom'} />
        )}
      >
        {i18nTemplates}
      </Button>
    );
  };

  handleTemplateClick = (template) => {
    if (this.props.ckeditor) {
      if (this.props.ckeditor.mode === 'source') {
        this.props.ckeditor.setData(template.bodyPlain + this.props.ckeditor.getData());
      } else {
        this.props.ckeditor.insertHtml(template.bodyHtml);
      }
    }

    if (this.props.onTemplateClick) {
      this.props.onTemplateClick(template);
    }

    this.popupBag.hide();
  };

  isPlain = () => {
    return this.props.ckeditor && this.props.ckeditor.mode === 'source';
  };

  content = (popupBag) => {
    this.popupBag = popupBag;

    return (
      <TemplatesList
        isPlain={this.isPlain}
        templates={this.props.templates}
        onTemplateClick={this.handleTemplateClick}
        onKeyUp={(e) => {
          switch (e.key) {
            case 'Escape':
              e.nativeEvent.stopImmediatePropagation();
              e.preventDefault();
              this.popupBag.hide();
              break;
            default:
          }
        }}
      />
    );
  };

  render() {
    return (
      <ImperativePopup
        className={cx(css.popup, css.popup_bottom)}
        toggle={this.toggle}
        content={this.content}
        scope={this.props.container}
        directions={['bottom-right']}
        zIndex={100}
      />
    );
  }
}

Templates.propTypes = {
  className: PropTypes.string,
  container: PropTypes.instanceOf(Object),
  size: PropTypes.string,
};

Templates.defaultProps = {
  className: undefined,
  container: undefined,
  size: 's',
};
