import * as React from 'react';
import Button, { ButtonViewProp } from '@crm/components/dist/lego2/Button';
import { LegoSizeProp } from '@crm/components/dist/lego2/types';
import ToolTip from 'lego/components/ToolTip';
import { FormattedDate } from 'react-intl';
import Icon from 'lego/components/Icon';
import { Timer as ITimer } from './types';
import Timers from './Timers';
import css from './TimersSmallWidget.scss';
import { Modal } from '../Form';

const iconProvider = (iconCls: string) => (
  <Icon className={iconCls} pack={Icon.PACKS.MATERIAL} icon="alarm" />
);

interface Props {
  items: ITimer[];
  size?: LegoSizeProp;
  view?: ButtonViewProp;
  disabled?: boolean;
  onRemove: (id: number) => {};
  onAdd: (timer: ITimer) => {};
  open: (args: unknown) => {};
}

class TimersSmallWidget extends React.Component<Props> {
  public static defaultProps = {
    view: 'pseudo',
    items: undefined,
  };

  private getCountItems = () => {
    const { items } = this.props;

    return (Array.isArray(items) && items.length) || 0;
  };

  private handleOpen = (initialValues?: unknown) => {
    const { open, onAdd } = this.props;

    open({ initialValues, isNew: !initialValues, onSubmit: onAdd });
  };

  private renderButtonInner = (): React.ReactNode => {
    const { items } = this.props;
    if (!items || !items.length) {
      return undefined;
    }

    const nearestStartDate = items[0].startDate;
    return (
      <FormattedDate
        value={new Date(nearestStartDate)}
        year="numeric"
        month="numeric"
        day="numeric"
        hour="numeric"
        minute="numeric"
      />
    );
  };

  public render(): React.ReactFragment {
    const { items, size, disabled, view, onRemove, onAdd } = this.props;

    const length = this.getCountItems();

    return (
      <ToolTip
        tooltip={
          Boolean(length) && (
            <Timers
              className={css.b__list}
              onRemove={onRemove}
              onAdd={onAdd}
              disabled={disabled}
              items={items}
            />
          )
        }
        hideDelay={100}
        directions={['bottom-center']}
        zIndexGroupLevel={4}
      >
        <Button
          onClick={this.handleOpen}
          size={size}
          view={view}
          iconLeft={length ? iconProvider : undefined}
          icon={length ? undefined : iconProvider}
          title={length ? 'Таймеры' : 'Создать новый таймер'}
          disabled={!length && disabled}
        >
          {this.renderButtonInner()}
        </Button>
      </ToolTip>
    );
  }
}

export default Modal.withOpen(TimersSmallWidget);
