import Promise from 'bluebird';
import { get, jsonApiCall } from 'api/common';
import { change } from 'redux-form';
import api from 'api';
import { CALL_MAIL_NAME } from 'modules/phone/constants';
import { push } from 'modules/notificationsUI/actions';
import { NAME } from './constants';
import {
  RESET_UPSALE,
  TOGGLE_FACTOR,
  UPSALE_DATA_REQUEST,
  UPSALE_DATA_RECEIVE,
  UPSALE_DATA_FAILED,
  UPSALE_CAMPAIGN_REQUEST,
  UPSALE_CAMPAIGN_RECEIVE,
  UPSALE_UPDATE_FILES_REQUEST,
  UPSALE_UPDATE_FILES_FINISH,
  UPSALE_SAVE_REQUEST,
  UPSALE_SAVE_RECEIVE,
  UPSALE_SAVE_DRAFT_REQUEST,
  UPSALE_SAVE_DRAFT_RECEIVE,
  UPSALE_SAVE_DRAFT_FAIL,
  UPSALE_REPLACE_STATE,
  UPSALE_SAVE_REQUEST_FINISH,
} from './actionTypes';
import {
  getUpsaleFactorsSelected,
  getUpsaleFactorsForOffer,
  getSelectedFactors,
  getUpsaleState,
  getUpsaleDraftId,
} from './selectors';
import { behaviors } from './reducers';
import SelectedFactors from './utils/SelectedFactors';
import { retryValidateRequest } from './utils/retryValidateRequest';

export const toggleFactor = ({ accountId, factorId, campaignId, factorSource }) => ({
  type: TOGGLE_FACTOR,
  payload: { factorId, campaignId, factorSource },
  meta: { name: accountId },
});

export const actionsMailFileUpdateLinks = (...args) => (dispatch) => {
  if (args[2] === '[]' && args[3] === '[]') {
    return;
  }
  dispatch({ type: UPSALE_UPDATE_FILES_REQUEST });
  return api.actions.mail.file
    .updateLinks(...args)
    .then((data = {}) => {
      dispatch(change(CALL_MAIL_NAME, 'files', data.files));
      dispatch(change(CALL_MAIL_NAME, 'containerId', data.containerId));
      dispatch({ type: UPSALE_UPDATE_FILES_FINISH });
      return data;
    })
    .catch((e) => {
      dispatch({ type: UPSALE_UPDATE_FILES_FINISH });
      throw e;
    });
};

export const resetUpsale = () => ({ type: RESET_UPSALE });

export const loadCampaign = (accountId, campaignId, chatId, issueId, mode) => (dispatch) => {
  dispatch({ type: UPSALE_CAMPAIGN_REQUEST });
  return get({
    url: '/view/upsale/campaign',
    data: { accountId, campaignId, chatId, issueId, mode },
    global: false,
  }).then((data) => {
    dispatch({
      type: UPSALE_CAMPAIGN_RECEIVE,
      data,
    });

    return data;
  });
};

export const loadFactors = ({ accountId, campaignId, chatId, issueId, mode, ycCallId }) => (
  dispatch,
) => {
  dispatch({ type: UPSALE_DATA_REQUEST });
  return get({
    url: '/view/upsale',
    data: { accountId, campaignId, chatId, issueId, mode, ycCallId },
    global: false,
  }).then((data) => {
    if (data.warnMessage) {
      dispatch(push({ theme: 'warn', text: data.warnMessage }));
      dispatch({ type: UPSALE_DATA_FAILED });
    } else {
      const formattedData = { ...data };
      formattedData.selectedFactors = SelectedFactors.backendToFrontend(
        formattedData.selectedFactors,
      );

      if (data.infoMessage) {
        dispatch(push({ theme: 'warn', text: data.infoMessage }));
      }

      dispatch({ type: UPSALE_DATA_RECEIVE, data: formattedData });
    }

    return data;
  });
};

export const loadAllFactors = ({ accountId, campaignId, chatId, issueId, mode, ycCallId }) => (
  dispatch,
) => {
  return new Promise((resolve, reject, onCancel) => {
    let campaignPromise;
    if (campaignId) {
      campaignPromise = dispatch(loadCampaign(accountId, campaignId, chatId, issueId, mode));
    }
    const factorsPromise = dispatch(
      loadFactors({ accountId, campaignId, chatId, issueId, mode, ycCallId }),
    );

    const campaignCancel = () => {
      if (campaignPromise) {
        campaignPromise.cancel();
      }
    };

    resolve(
      factorsPromise.then((data) => {
        campaignCancel();

        return data;
      }),
    );

    onCancel(() => {
      campaignCancel();
      factorsPromise.cancel();
    });
  });
};

export const saveFactors = (payload) => (dispatch, getState) => {
  const { accountId, chatId, issueId, cancelled = false } = payload;

  const state = getState();

  const offeredFactors = getUpsaleFactorsSelected(state) || [];
  const factorsForOffer = getUpsaleFactorsForOffer(state) || [];
  const upsaleDraftId = getUpsaleDraftId(state) || undefined;

  function request(needValidate = true) {
    dispatch({ type: UPSALE_SAVE_REQUEST });
    return jsonApiCall({
      url: '/actions/upsale/save',
      data: {
        accountId,
        chatId,
        offeredFactors,
        factorsForOffer,
        cancelled,
        upsaleDraftId,
        issueId,
        needValidate,
      },
      global: false,
    })
      .then((data) => {
        return retryValidateRequest(data, request);
      })
      .then((data) => {
        if (data != null) {
          dispatch({
            type: UPSALE_SAVE_RECEIVE,
            data,
          });
        }

        return data;
      })
      .finally(() => {
        dispatch({ type: UPSALE_SAVE_REQUEST_FINISH });
      });
  }

  return request();
};

export const saveDraftFactors = (args) => (dispatch, getState) => {
  const { accountId, chatId, issueId, factorId, campaignId, factorSource } = args;

  function request(needValidate, state) {
    const newState = {
      [NAME]: behaviors[TOGGLE_FACTOR](getUpsaleState(state), {
        payload: { factorId, campaignId, factorSource },
      }),
    };

    const offeredFactors = getUpsaleFactorsSelected(newState) || [];
    const factorsForOffer = getUpsaleFactorsForOffer(newState) || [];

    dispatch({ type: UPSALE_SAVE_DRAFT_REQUEST });
    return jsonApiCall({
      url: '/actions/upsale/saveDraft',
      data: {
        accountId,
        chatId,
        issueId,
        offeredFactors,
        factorsForOffer,
        selectedFactors: SelectedFactors.frontendToBackend(getSelectedFactors(newState)),
        needValidate,
      },
      global: false,
    })
      .then((data) => {
        dispatch({ type: UPSALE_SAVE_DRAFT_RECEIVE });
        return data;
      })
      .then((data) => {
        return retryValidateRequest(data, request, state);
      })
      .then((data) => {
        if (data != null) {
          dispatch({ type: UPSALE_REPLACE_STATE, payload: newState[NAME] });
        }

        return data;
      })
      .catch(() => {
        dispatch({ type: UPSALE_SAVE_DRAFT_FAIL });
      });
  }

  const state = getState();
  return request(true, state);
};

export default {
  toggleFactor,
  actionsMailFileUpdateLinks,
  resetUpsale,
  loadFactors,
  loadCampaign,
  loadAllFactors,
};
