import React from 'react';
import PropTypes from 'prop-types';
import CopyButton from 'lego/components/CopyButton';
import cx from 'classnames';
import css from './styles.modules.scss';

export default class Factor extends React.Component {
  constructor(props) {
    super(props);

    const { factorCounters, campaignId } = this.props;

    this.state = {
      enabled: this.props.getDefaultFactorState(factorCounters.id, campaignId),
    };
  }

  componentDidUpdate(prevProps) {
    if (this.props.saveRequestHash !== prevProps.saveRequestHash && this.state.enabled) {
      const { factorCounters, campaignId } = this.props;

      this.setState({ enabled: this.props.getDefaultFactorState(factorCounters.id, campaignId) });
    }
  }

  toggle = () => {
    const { enabled } = this.state;

    this.setState({ enabled: !enabled });
  };

  handleFactorClick = () => {
    const { factorCounters, campaignId, onFactorClick } = this.props;

    onFactorClick(factorCounters, campaignId, this.toggle);
  };

  emptyHandle = () => {};

  render() {
    const { factor, factorCounters, isSupportCopyToClipboard, clickDisabled } = this.props;

    if (!factor) {
      return null;
    }
    const combinedFactor = {
      ...factor,
      ...factorCounters,
    };

    const { active, color } = combinedFactor;

    const factorStyle = cx(
      css.factor,
      clickDisabled ? css.factor_disabledClick : '',
      {
        [css.factor_offered__emphasized]: combinedFactor.factorSource === 'Direct',
        [css.factor_hover]: active,
        [css.factor_disabled]: !active,
      },
      { [css.factor_selected]: this.state.enabled },
    );

    const style = { background: color };
    const onClickHandle = clickDisabled ? this.emptyHandle : this.handleFactorClick;

    return (
      <span className={css.factorWrap}>
        <button
          type="button"
          style={style}
          className={factorStyle}
          onClick={onClickHandle}
          title={combinedFactor.scriptPlain}
          disabled={!active}
        >
          <span className={css.factor__text}>{combinedFactor.name}&nbsp;</span>
          {factorCounters.percent != null && (
            <span className={css.factor__percent}>
              <i>
                {factorCounters.percent}%&nbsp;{factorCounters.campaignCount}РК
              </i>
            </span>
          )}
        </button>
        {isSupportCopyToClipboard && (
          <CopyButton
            style={style}
            size="xs"
            view="clear"
            className={cx(factorStyle, css.factor_copy)}
            copy={combinedFactor.mailPlain}
            title="Скопировать"
          />
        )}
      </span>
    );
  }
}

Factor.propTypes = {
  factor: PropTypes.instanceOf(Object).isRequired,
  factorCounters: PropTypes.instanceOf(Object).isRequired,
  isSupportCopyToClipboard: PropTypes.bool,
  saveRequestHash: PropTypes.number,
  getDefaultFactorState: PropTypes.func.isRequired,
  onFactorClick: PropTypes.func.isRequired,
  campaignId: PropTypes.number,
  clickDisabled: PropTypes.bool,
};

Factor.defaultProps = {
  isSupportCopyToClipboard: false,
  saveRequestHash: undefined,
  campaignId: undefined,
  clickDisabled: false,
};
