import React from 'react';
import PropTypes from 'prop-types';
import applyMixins from 'utils/applyMixins';
import Loadable from 'utils/Loadable';
import $ from 'jquery';
import get from 'lodash/get';
import Editor from '../../utils/editor';
import { WITHOUT_CAMPAIGN_ID } from '../../constants';
import Layout from './Layout';

const getCampaignsArray = (campignsObject = {}) =>
  Object.keys(campignsObject).filter((key) => key !== 'length' && key !== WITHOUT_CAMPAIGN_ID);

const FactorChangeEventType = {
  Add: 'add',
  Remove: 'remove',
  Change: 'change',
};

export class LayoutContainer extends React.Component {
  constructor(props) {
    super(props);
    const e = new Editor(
      'body',
      'spFactor',
      () => e.clean(),
      () => e.switchMode(this.generateMailTemplate()),
    );
    this.editor = e;
  }

  componentDidMount() {
    this.load();
  }

  componentDidUpdate(previousProps) {
    if (
      this.props.accountId !== previousProps.accountId ||
      this.props.campaignId !== previousProps.campaignId
    ) {
      this.props.resetUpsale();
      this.load();
    }

    this.updateFactorFilesAndTemplatesIfNeeded(previousProps);
  }

  componentWillUnmount() {
    this.loadCancel();
    this.props.resetUpsale();
  }

  updateFactorFilesAndTemplatesIfNeeded(previousProps) {
    if (
      this.props.upsale !== previousProps.upsale &&
      !this.props.upsale.isUpdatingFiles &&
      this.props.upsale.lastFactor &&
      this.props.upsale.lastFactor !== previousProps.upsale.lastFactor
    ) {
      this.updateFactorFilesAndTemplates(this.props.upsale.lastFactor.id);
    }
  }

  /**
   * @param factor {Object}
   * @returns {String}
   * @private
   */
  getFactorTemplate(factor) {
    const suffix = this.editor.isPlain() ? 'Plain' : 'Html';
    const proKey = `mail${suffix}`;

    return factor[proKey];
  }

  getFactorTemplateWrap(factor) {
    const output = this.getFactorTemplate(factor);

    // return output;

    if (this.editor.isPlain()) {
      return output;
    }
    return output ? `<div data-crm-factor="${factor.code}">${output}</div>` : '';
  }

  getDefaultFactorState = (factorId, campaignId) => {
    return get(
      this.props.upsale,
      `selectedFactors.${factorId}.${campaignId || WITHOUT_CAMPAIGN_ID}`,
      false,
    );
  };

  updateFactorFilesAndTemplates = async (factorId) => {
    const { onFactorChange, factorStream } = this.props;
    const { selectedFactors } = this.props.upsale;
    const isSelected = Boolean(selectedFactors[factorId]);

    const factor = this.props.factors[factorId];

    const templateText = factor.mailPlain?.replace('{{campaignId}}', factor.campaignId ?? '') ?? '';

    if (factorStream) {
      factorStream.next({
        type: this.props.upsale.lastFactor.status ?? FactorChangeEventType.change,
        payload: { factorId, templateText, files: factor.files },
      });
    }

    if (this.editor.instance && (await this.editor.isExists())) {
      if (this.editor.isPlain()) {
        this.editor.insert(this.generateMailTemplate());
      } else {
        const factorHtml = !isSelected ? null : this.generateFactorTemplate(factorId, factor);
        this.editor.insertFactor(factorHtml, factor.code);
      }

      if (onFactorChange) {
        this.updateFileLinks2(factor);
      } else {
        this.updateFileLinks(factor);
      }
    }
  };

  handleFactorClick = (factor, campaignId, toggle) => {
    const {
      isUpdatingFiles, // eslint-disable-line
      upsaleFactorsSelected, // eslint-disable-line
      maxFactorsCount, // eslint-disable-line
      toggleFactor, // eslint-disable-line
      saveDraftFactors, // eslint-disable-line
      isSupportSaveDraft,
    } = this.props;

    if (isUpdatingFiles) {
      return;
    }

    if (
      // eslint-disable-next-line
      upsaleFactorsSelected.length >= maxFactorsCount &&
      // eslint-disable-next-line
      upsaleFactorsSelected.findIndex((f) => f.id === factor.id) < 0
    ) {
      alert(`Нельзя выбрать более ${maxFactorsCount} факторов!`); // eslint-disable-line
      return;
    }

    if (isSupportSaveDraft) {
      saveDraftFactors(factor, campaignId).then((data) => {
        if (data != null) {
          toggle();
        }

        return data;
      });
    } else {
      toggle();
      toggleFactor(factor, campaignId);
    }
  };

  /**
   * @returns {String}
   * @private
   */
  generateMailTemplate() {
    const content = [];

    const { selectedFactors } = this.props.upsale;

    Object.keys(selectedFactors || {}).forEach((factorId) => {
      const factor = this.props.factors[factorId];
      const campaigns = getCampaignsArray(selectedFactors[factorId]);

      let template = this.getFactorTemplateWrap(factor);

      if (!template) {
        return;
      }

      template = template.replace(
        '{{campaignId}}',
        campaigns.length > 0 ? `(РК: ${campaigns.join(', ')})` : '',
      );

      content.push(template);
    });

    const newLine = this.editor.isPlain() ? '\n\n\n' : '<br><br><br>';

    return content.join(newLine);
  }

  generateFactorTemplate(factorId, factor) {
    let template = this.editor.getFactorHtml(factor.code);
    const factorsInfo = this.props.upsale;
    const { selectedFactors } = factorsInfo;

    const campaigns = getCampaignsArray(selectedFactors[factorId]);

    const rkExists = campaigns && campaigns.length > 0 && campaigns[0];
    if (!template) {
      template = this.getFactorTemplateWrap(factor);
      if (!template) {
        return '';
      }

      template = template.replace(
        '{{campaignId}}',
        rkExists
          ? `<span data-crm-campaigns>(РК: ${campaigns.join(', ')})</span>`
          : '<span data-crm-campaigns> </span>',
      );
    } else {
      const $template = $(template);
      const $campaignsText = $template.find('span[data-crm-campaigns]');
      let text = '';
      if (rkExists) {
        text = `(РК: ${campaigns.join(', ')})`;
      }
      $campaignsText.text(text);

      template = $template.html();
      $template.remove();

      if (!campaigns || campaigns.length === 0) {
        return '';
      }
    }

    return template;
  }

  updateFileLinks(factor) {
    if (!factor.files) {
      return;
    }

    const { mailFiles, mailContainerId, upsale, updateFactorFileLinks, mailId } = this.props;

    const filesToLink = [];
    const filesToUnlink = [];

    let operationArray = null;
    if (upsale.lastFactor.status === 'add') {
      operationArray = filesToLink;
    } else if (upsale.lastFactor.status === 'remove') {
      operationArray = filesToUnlink;
    } else {
      return;
    }

    for (let i = 0; i < factor.files.length; i += 1) {
      const file = factor.files[i];
      operationArray.push(file);
    }

    const nameMapper = (mailFiles || []).reduce((ac, file) => {
      /* eslint-disable no-param-reassign */
      if (!ac[file.name]) {
        ac[file.name] = [];
      }
      ac[file.name].push(file.id);

      /* eslint-enable */

      return ac;
    }, {});

    const filteredFilesToUnlink = Object.keys(
      filesToUnlink.reduce((ac, file) => {
        /* eslint-disable no-param-reassign */
        ac[file.name] = true;
        /* eslint-enable */
        return ac;
      }, {}),
    )
      .map((name) => nameMapper[name])
      .filter((arr) => arr)
      .reduce((ac, arr) => ac.concat(arr), []);

    if (!filesToLink.length && !filteredFilesToUnlink.length) {
      return;
    }

    updateFactorFileLinks(
      mailId,
      mailContainerId,
      JSON.stringify(filteredFilesToUnlink),
      JSON.stringify(filesToLink.map((file) => file.id)),
    );
  }

  updateFileLinks2(factor) {
    const { onFactorChange, upsale } = this.props;
    if (upsale.lastFactor.status === 'add') {
      onFactorChange(FactorChangeEventType.Add, factor);
    } else if (upsale.lastFactor.status === 'remove') {
      onFactorChange(FactorChangeEventType.Remove, factor);
    }
  }

  reloadUpsale() {
    this.props.resetUpsale();
    this.load();
  }

  loadRequest() {
    return this.props.loadAllFactors({
      accountId: this.props.accountId,
      campaignId: this.props.campaignId,
      chatId: this.props.chatId,
      issueId: this.props.issueId,
      mode: this.props.mode,
    });
  }

  render() {
    return (
      <Layout
        {...this.props}
        getDefaultFactorState={this.getDefaultFactorState}
        onFactorClick={this.handleFactorClick}
      />
    );
  }
}

LayoutContainer.propTypes = {
  mailFiles: PropTypes.instanceOf(Array),
  mailContainerId: PropTypes.number,
  updateFactorFileLinks: PropTypes.func.isRequired,
  resetUpsale: PropTypes.func.isRequired,
  mailId: PropTypes.number,
  accountId: PropTypes.number.isRequired,
  upsale: PropTypes.instanceOf(Object),
  factors: PropTypes.instanceOf(Object),
  campaignId: PropTypes.number,
  loadAllFactors: PropTypes.func.isRequired,
  isLoadingUpsale: PropTypes.bool.isRequired,
  isSupportSaveDraft: PropTypes.bool,
  chatId: PropTypes.string,
  issueId: PropTypes.number,
  mode: PropTypes.string,
  factorStream: PropTypes.instanceOf(Object),
};

LayoutContainer.defaultProps = {
  mailId: null,
  mailContainerId: undefined,
  campaignId: null,
  factors: [],
  upsale: null,
  mailFiles: undefined,
  isSupportSaveDraft: false,
  chatId: undefined,
  issueId: undefined,
  mode: undefined,
};

applyMixins(LayoutContainer, [Loadable]);
