import $ from 'jquery';
import { getCKEDITOR } from 'components/RichHtmlEditor';

class Editor {
  static PLAIN_MARKER = '\u0000';
  instanceId;
  /**
   * @param {String} id
   * @param {String} containerClass
   * @param {Function} onBeforeSetMode
   * @param {Function} onMode
   */
  constructor(id, containerClass, onBeforeSetMode, onMode) {
    this.containerClass = containerClass;
    getCKEDITOR().then((CKEDITOR) => {
      const editor = CKEDITOR.instances[id];
      this.instanceId = id;
      if (editor) {
        this.initializeEditorInstance(editor, onBeforeSetMode, onMode);
      }
      CKEDITOR.on('instanceReady', (e) => {
        if (e.editor.name === id) {
          this.initializeEditorInstance(e.editor, onBeforeSetMode, onMode);
        }
      });
    });
  }

  initializeEditorInstance(editor, onBeforeSetMode, onMode) {
    this.instance = editor;
    this.instance.on('beforeSetMode', onBeforeSetMode);
    this.instance.on('mode', onMode);
  }

  async isExists() {
    const CKEDITOR = await getCKEDITOR();
    return Boolean(CKEDITOR.instances[this.instanceId]);
  }

  /**
   * @param {String} content
   */
  insert(content, noNewLine = false) {
    if (this.isPlain()) {
      this.insertPlainText(content, false, noNewLine);
    } else {
      this.insertHtml(content, false, noNewLine);
    }
  }

  switchMode(content, noNewLine = false) {
    if (this.isPlain()) {
      this.insertPlainText(content, false, noNewLine);
    } else {
      this.setHtml(content, false, noNewLine);
    }
  }

  getContainer(domElementSelector) {
    if (!this.instance || !this.instance.editable || typeof this.instance.editable !== 'function') {
      return undefined;
    }
    const editable = this.instance.editable();
    if (!editable || !editable.$) {
      return undefined;
    }
    const editableJq = $(editable.$);
    const container = editableJq.find(domElementSelector);

    return container;
  }

  /**
   * Remove factor by data attribute (data-crm-factor="{factor.code}") from html
   */
  removeFactor(factorCode) {
    const container = this.getContainer(`div[data-crm-factor="${factorCode}"]`);
    if (!container) {
      return;
    }
    container.remove();
    this.instance.focus();
  }

  removeAllFactors() {
    const editableJq = $(`<div>${this.instance.switchmodePreviousData}</div>`);
    editableJq.find('div[data-crm-factor]').each(() => $(this).remove());

    this.instance.switchmodePreviousData = editableJq.html();
  }

  /**
   * get factor html by data attribute (data-crm-factor="{factor.code}")
   */
  getFactorHtml(factorCode) {
    const container = this.getContainer(`div[data-crm-factor="${factorCode}"]`);
    return (container && container[0] && container[0].outerHTML) || '';
  }

  setFactorHtml(factorCode, html) {
    const container = this.getContainer(`div[data-crm-factor="${factorCode}"]`);
    if (!container) {
      return;
    }
    container.html(html);
  }

  insertFactor(factorHtml, factorCode) {
    if (!factorHtml) {
      this.removeFactor(factorCode);
      return;
    }

    const container = this.getContainer(`div[data-crm-factor="${factorCode}"]`);
    if (container && container.length) {
      container.html(factorHtml);
    } else {
      const newFactorHtml = `<div data-crm-factor="${factorCode}" class="crm-editor-factor">${factorHtml}</div><div><br ></div>`;

      const factorContainer = this.getContainer('div[data-crm-macro="factors"]');
      if (factorContainer && factorContainer.length) {
        factorContainer.append(newFactorHtml);
        this.instance.fire('change');
      } else {
        const sel = this.instance.getSelection();
        const element = sel && sel.getStartElement();

        let $factor = null;
        if (element) {
          $factor = $(element.$).closest('div[data-crm-factor]');
        }

        if ($factor && $factor[0]) {
          $factor.after(newFactorHtml);
          this.instance.fire('change');
        } else {
          this.insert(newFactorHtml);
        }
      }
    }
  }

  /**
   * Removes plain text and html containers
   */
  clean() {
    if (this.isPlain()) {
      this.insertPlainText(null, true);
    } else {
      /* this.insertHtml(null, true) */
      this.removeAllFactors();
    }
  }

  /**
   * @returns {Boolean}
   */
  isPlain() {
    return this.instance.mode === 'source';
  }

  /**
   * @returns {String}
   */
  getContainerData() {
    if (this.isPlain()) {
      const data = this.instance.getData();
      const [start, end] = Editor.getPlainTextContainer(data);

      return data.substring(start + 1, end);
    }
    return this.getHtmlContainer().html();
  }

  /**
   * @param text {String}
   * @returns {String}
   */
  cleanPlainTextMarker = (text) => text.replace(new RegExp(Editor.PLAIN_MARKER, 'gim'), '');

  /**
   * @param html {String}
   * @returns {String}
   */
  cleanDataAttr(html) {
    if (html === '') {
      return '';
    }

    const $html = $(`<div>${html}</div>`);
    $html.find('[data-crm-campaigns]').each(() => $(this)[0].removeAttribute('data-crm-campaigns'));
    $html.find('[data-crm-factor]').each(() =>
      $(this)
        .removeClass('crm-editor-factor')[0]
        .removeAttribute('data-crm-factor'),
    );

    const output = $html.html() || '';
    $html.remove();

    return output;
  }

  cleanOutput(html) {
    let output = this.cleanPlainTextMarker(html);
    if (!this.isPlain()) {
      output = this.cleanDataAttr(output);
    }

    return output;
  }

  /**
   * @param fakeEditableJq
   * @returns {jQuery|null}
   * @private
   */
  getHtmlContainer(fakeEditableJq) {
    const selector = `.${this.containerClass}:first`;
    let container;
    if (fakeEditableJq) {
      container = fakeEditableJq.find(selector);
    } else {
      container = this.getContainer(selector);
    }
    if (container) {
      const inBlockquote = container.closest('blockquote').length !== 0;
      return inBlockquote || container.length === 0 ? null : container;
    }
    return null;
  }

  /**
   * @param data {String}
   * @returns {[Number, Number]}
   * @private
   */
  static getPlainTextContainer(data) {
    const start = data.indexOf(Editor.PLAIN_MARKER);
    const end = data.lastIndexOf(Editor.PLAIN_MARKER);

    return [start, end];
  }

  /**
   * @param html {String}
   * @param remove {Boolean}
   * @param noNewLine
   * @private
   */
  insertHtml(html, remove = false) {
    // , noNewLine = false
    let fakeEditableJq;

    if (this.instance.switchmodePreviousData) {
      fakeEditableJq = $('<div />');
      fakeEditableJq.html(this.instance.switchmodePreviousData);
    }
    const container = this.getHtmlContainer(fakeEditableJq);
    const newContainer = container === null;
    if (remove) {
      if (!newContainer) {
        container.remove();

        if (fakeEditableJq) {
          this.instance.switchmodePreviousData = fakeEditableJq.html();
        }
      }

      return;
    }
    this.instance.insertHtml(html);

    /*        if (!container) {
          const newLine = noNewLine ? '' : '<br>';
          this.instance.insertHtml(`${newLine}<div class="${this.containerClass}">${html}</div>`);
      } else {
          container.html(html);
      } */
  }

  setHtml(html, remove = false) {
    let fakeEditableJq;

    if (this.instance.switchmodePreviousData) {
      fakeEditableJq = $('<div />');
      fakeEditableJq.html(this.instance.switchmodePreviousData);
    }

    const container = this.getHtmlContainer(fakeEditableJq);
    const newContainer = container === null;

    if (remove) {
      if (!newContainer) {
        container.remove();

        if (fakeEditableJq) {
          this.instance.switchmodePreviousData = fakeEditableJq.html();
        }
      }

      return;
    }

    this.instance.setData(html + this.instance.getData());
  }

  /**
   * @param text {String}
   * @param remove {Boolean}
   * @param noNewLine {Boolean}
   * @private
   */
  insertPlainText(text, remove = false, noNewLine = false) {
    const oldData = this.instance.getData();
    const [start, end] = Editor.getPlainTextContainer(oldData);
    const newContainer = start === end;

    if (remove) {
      if (!newContainer) {
        if (this.instance.switchmodePreviousData) {
          this.instance.switchmodePreviousData =
            oldData.substring(0, start) + oldData.substring(end + 1);
        } else {
          this.instance.setData(oldData.substring(0, start) + oldData.substring(end + 1));
        }
      }

      return;
    }

    let newData;

    if (start === end) {
      const newLine = noNewLine ? '' : '\n';
      newData = `${newLine}${Editor.PLAIN_MARKER}${text}${Editor.PLAIN_MARKER}${newLine}${oldData}`;
    } else {
      newData = oldData.substring(0, start + 1) + text + oldData.substring(end);
    }

    this.instance.setData(newData);
  }
}

export default Editor;
