import { createSlice, PayloadAction } from '@reduxjs/toolkit';
import { UserStatusId } from 'types/UserStatusId';
import { UserStatus } from 'types/UserStatus';
import { NAME } from './Status.constants';
import {
  StatusStoreState,
  Dispatch,
  GetState,
  SliceState,
  Status,
  Flag,
  Problem,
} from './Status.types';
import requests from './Status.requests';

const initialState: SliceState = {
  isFetching: false,
  current: undefined,
  transitions: [],
  flags: [],
  problems: [],
};

const StatusSlice = createSlice({
  name: NAME,
  initialState,
  reducers: {
    setCurrent: (state, action: PayloadAction<Status | undefined>) => {
      return {
        ...state,
        current: action.payload,
      };
    },

    setTransitions: (state, action: PayloadAction<Status[]>) => {
      return {
        ...state,
        transitions: action.payload,
      };
    },

    setFlags: (state, action: PayloadAction<Flag[] | undefined>) => {
      return {
        ...state,
        flags: action.payload,
      };
    },

    setProblems: (state, action: PayloadAction<Problem[] | undefined>) => {
      return {
        ...state,
        problems: action.payload,
      };
    },

    startFetch: (state) => {
      return {
        ...state,
        isFetching: true,
      };
    },

    finishFetch: (state) => {
      return {
        ...state,
        isFetching: false,
      };
    },
  },
});

const syncActions = StatusSlice.actions;

export const selectors = {
  getState: (state: StatusStoreState): SliceState => state[NAME],
  getCurrentStatus: (state: StatusStoreState): Status | undefined => state[NAME].current,
  getTransitions: (state: StatusStoreState): Status[] => state[NAME].transitions,
  getFlags: (state: StatusStoreState): Flag[] | undefined => state[NAME].flags,
  getProblems: (state: StatusStoreState): Problem[] | undefined => state[NAME].problems,
  isFetching: (state: StatusStoreState): boolean => state[NAME].isFetching,
};

const getStatusAction = () => async (dispatch: Dispatch): Promise<void> => {
  dispatch(syncActions.startFetch());
  requests
    .getStatus()
    .then((response) => {
      const { status, transitions, flags, problems } = response;
      dispatch(syncActions.setTransitions(transitions));
      dispatch(syncActions.setCurrent(status));
      dispatch(syncActions.setFlags(flags));
      dispatch(syncActions.setProblems(problems));
    })
    .finally(() => {
      dispatch(syncActions.finishFetch());
    });
};

const setStatusAction = (data: {
  status?: Status;
  flags?: Pick<Flag, 'name' | 'value'>[];
  problems?: Pick<Problem, 'name'>[];
}) => async (dispatch: Dispatch, getState: GetState): Promise<void> => {
  const prevStatus = selectors.getCurrentStatus(getState());
  const { status, flags, problems } = data;

  dispatch(syncActions.startFetch());
  requests
    .setStatus({
      data: {
        id: status?.id,
        statusId: status?.id,
        flags,
        problems,
      },
      global: false,
    })
    .then((response) => {
      const { transitions, flags, problems } = response;

      dispatch(syncActions.setTransitions(transitions));
      dispatch(syncActions.setCurrent(response.status ? response.status : status));
      dispatch(syncActions.setFlags(flags));
      dispatch(syncActions.setProblems(problems));
    })
    .catch((error) => {
      console.error(error);
      dispatch(syncActions.setCurrent(prevStatus));
    })
    .finally(() => {
      dispatch(syncActions.finishFetch());
    });
};

const setStatusByStatusId = (statusId: UserStatusId) => (
  dispatch: Dispatch,
  getState: GetState,
) => {
  const prevStatus = selectors.getCurrentStatus(getState());

  if (statusId === prevStatus?.id) {
    return;
  }

  const transitions = selectors.getTransitions(getState());

  const transitionsMap = transitions.reduce((ac, item) => {
    ac.set(item.id, item);
    return ac;
  }, new Map<number, UserStatus>());

  const nextStatus = transitionsMap.get(statusId);

  if (!nextStatus) {
    return;
  }

  return dispatch(setStatusAction({ status: nextStatus }));
};

export const actions = {
  ...syncActions,
  setStatus: setStatusAction,
  setStatusByStatusId,
  getStatus: getStatusAction,
};

export default StatusSlice.reducer;
