import { useEffect } from 'react';
import { logger } from 'services/Logger';
import { CallKind } from '@yandex-telephony/ya-calls-webphone-sdk';
import { CallStatusLog } from './CallStatusLog';
import { CallStatusServiceComponentProps } from './CallStatusServiceComponent.types';
import { CallStatusService } from '../../services/CallStatusService';
import {
  WebphoneEventManager,
  WebphoneOutgoingEventKind,
} from '../../services/WebphoneEventManager';

const callStatusService = new CallStatusService();

export const CallStatusServiceComponent = (props: CallStatusServiceComponentProps) => {
  const { webphone } = props;
  useEffect(() => {
    if (!webphone) {
      return;
    }

    let activeCallId: string | undefined = undefined;

    logger.reportInfo(new CallStatusLog('use effect', { webphone: String(webphone) }));

    const manager = new WebphoneEventManager(webphone, {
      [WebphoneOutgoingEventKind.NewOutgoingCall]: (event) => {
        logger.reportInfo(
          new CallStatusLog('NewOutgoingCall', {
            callId: event.callId,
            activeCallId: String(activeCallId),
          }),
        );
        callStatusService.startCall(event.callId);
        activeCallId = event.callId;
      },
      [WebphoneOutgoingEventKind.EstablishedCall]: (event) => {
        logger.reportInfo(
          new CallStatusLog('EstablishedCall', {
            callId: event.callId,
            activeCallId: String(activeCallId),
          }),
        );
        if (event.callKind === CallKind.Incoming) {
          callStatusService.startCall(event.callId);
          activeCallId = event.callId;
        }
      },
      [WebphoneOutgoingEventKind.CallEnd]: (event) => {
        logger.reportInfo(
          new CallStatusLog('CallEnd', {
            callId: event.callId,
            activeCallId: String(activeCallId),
            canBeError: String(event.callId !== activeCallId),
          }),
        );
        if (event.callId === activeCallId) {
          callStatusService.endCall(event.callId);
        }
      },
    });

    return () => {
      manager.destroy();
    };
  }, [webphone]);

  return null;
};
