import React, { FC, useContext, useCallback } from 'react';
import Link from '@crm/components/dist/lego2/Link';
import { useDispatch } from 'react-redux';
import { push } from 'modules/notificationsUI';
import { WebphoneError } from '@yandex-telephony/ya-calls-webphone-sdk';
import { webphoneContext } from '../WebphoneContext/WebphoneContext';
import { callLockContext } from '../CallLockContext';
import webphoneStateContext from '../WebphoneStateContext/webphoneStateContext';
import { CallToProps } from './CallTo.types';
import { errors } from './CallTo.config';

export const CallTo: FC<CallToProps> = ({
  callee,
  caller,
  associatedUnitId,
  as: Component = Link,
  onCallInit,
  onCallError,
  callAvailableService,
  callMetaService,
  children,
  ...other
}) => {
  const webphone = useContext(webphoneContext);
  const webphoneState = useContext(webphoneStateContext);
  const callLockService = useContext(callLockContext);
  const dispatch = useDispatch();

  const onCall = useCallback(async () => {
    /* https://st.yandex-team.ru/CRM-13131#60d1d0618ed18652b0d6d917 */
    if (callee === '-') {
      throw new Error(errors.wrongNumber);
    }

    if (!webphone) {
      throw new Error(errors.webphoneNotInit);
    }

    if (callLockService.isLock) {
      throw new Error(errors.creatingCall);
    }

    try {
      callLockService.lock();

      const canCallData = await callAvailableService.canCall();

      if (!canCallData.canCall) {
        dispatch(push({ level: 'warning', text: canCallData.message }));
        callLockService.release();
        return;
      }

      const meta = await callMetaService.getCallMeta();

      if (onCallInit) {
        onCallInit({ webphone, meta });
      }

      const result = await webphone.createCall(meta.caller, callee, meta.associatedUnitId);
      return result;
    } catch (error) {
      /* https://st.yandex-team.ru/CRM-12295 */
      if (error !== WebphoneError.Internal) {
        callLockService.release();
        if (onCallError) {
          onCallError(error);
        }
      }

      throw error;
    }
  }, [
    webphone,
    callLockService,
    callAvailableService,
    callMetaService,
    onCallInit,
    callee,
    dispatch,
    onCallError,
  ]);

  return (
    <Component disabled={Boolean(webphoneState.call)} onClick={onCall} {...other}>
      {children || callee}
    </Component>
  );
};
