import React, { createRef } from 'react';
import { WebphoneWidget, WebphoneWidgetEnvironment } from '@yandex-telephony/ya-calls-webphone-sdk';
import { BackendEnv } from 'hoc/withBackendEnv';
import { config } from 'services/Config';
import { runIfTabVisibleOnce, Unsubscribe } from 'utils/runIfTabVisibleOnce';
import { webphoneIssueRedirectService } from 'modules/webphone/services/WebphoneIssueRedirectService';
import { WebphoneIncomingCallEndService } from 'modules/webphone/services/WebphoneIncomingCallEndService';
import { terminalIdService } from 'services/TerminalIdService';
import { webphoneService } from 'services/WebphoneService';
import { WebphoneProps } from './Webphone.types';
import { WebphoneLogger } from '../../services/WebphoneLogger';
import { WebphoneFaviconService } from '../../services/WebphoneFaviconService';

export class Webphone extends React.Component<WebphoneProps> {
  public static readonly EnvMap = {
    [BackendEnv.Test]: WebphoneWidgetEnvironment.Preproduction,
    [BackendEnv.Prod]: WebphoneWidgetEnvironment.Production,
  };

  public isInit = false;

  private nodeRef = createRef<HTMLSpanElement>();

  private unsubscribeCreateWebphone: Unsubscribe;

  private webphone?: WebphoneWidget;
  private webphoneLogger?: WebphoneLogger;
  private webphoneFaviconService?: WebphoneFaviconService;
  private webphoneIncomingCallEndService?: WebphoneIncomingCallEndService;

  public componentDidMount(): void {
    this.unsubscribeCreateWebphone = runIfTabVisibleOnce(this.createWebphone);
  }

  public shouldComponentUpdate(): boolean {
    return false;
  }

  public componentWillUnmount(): void {
    this.webphone?.destruct();
    this.webphoneLogger?.destroy();
    this.webphoneFaviconService?.destroy();
    this.unsubscribeCreateWebphone();

    if (config.value.features.useYaCalls) {
      this.webphoneIncomingCallEndService?.destroy();
    }
  }

  public render() {
    return <span ref={this.nodeRef} />;
  }

  private createWebphone = () => {
    this.webphone = new WebphoneWidget(this.nodeRef.current!, {
      env: Webphone.EnvMap[this.props.env],
      hideStatusSwitch: true,
      initTimeoutInMs: 60000,
      isOpen: false,
      onSwitchVisibility: this.handleSwitchVisibility,
    });

    this.webphone
      .init()
      .then((data) => {
        terminalIdService.value = data.terminalId;
      })
      .catch((error) => {
        console.warn(error);
      });

    webphoneService.setWebphone(this.webphone);

    this.webphoneLogger = new WebphoneLogger(this.webphone);
    this.webphoneFaviconService = new WebphoneFaviconService(this.webphone);

    if (config.value.features.useYaCalls) {
      webphoneIssueRedirectService.setWebphone(this.webphone);
      this.webphoneIncomingCallEndService = new WebphoneIncomingCallEndService();
    }

    if (this.props.onInit) {
      this.props.onInit(this.webphone);
    }
  };

  private handleSwitchVisibility = (isVisible: boolean) => {
    if (this.props.onSwitchVisibility) {
      this.props.onSwitchVisibility(isVisible);
    }
  };
}
