import {
  CallKind,
  WebphoneCallEndEvent,
  WebphoneEstablishedCallEvent,
  WebphoneOutgoingEventKind,
  WebphoneWidget,
} from '@yandex-telephony/ya-calls-webphone-sdk';
import { Subject } from 'rxjs';
import { jsonApiCall } from 'api/common';
import { WebphoneEventManager } from './WebphoneEventManager';
import {
  AcceptCallRequest,
  AcceptCallResponse,
  IncomingCallEndData,
  OnCallAcceptSubscription,
  OnCallEndSubscription,
  AcceptCallData,
} from './WebphoneIncomingCallService.types';
import { establishedCallEventListener } from '../utils/establishedCallEventListener';
import { webphoneStateManager } from './WebphoneStateManager';

export class WebphoneIncomingCallService {
  private manager: WebphoneEventManager;

  private hasActiveIncomingCall = false;

  private webphone: WebphoneWidget;

  public callEndSubject = new Subject<IncomingCallEndData>();

  private acceptCallSubject = new Subject<AcceptCallData>();

  public acceptCallData?: AcceptCallData;

  public setWebphone(webphone: WebphoneWidget) {
    this.webphone = webphone;
    this.init();
  }

  public destroy() {
    this.manager.destroy();
  }

  private init() {
    this.manager = new WebphoneEventManager(this.webphone, {
      [WebphoneOutgoingEventKind.EstablishedCall]: this.handleEstablishedCall,
      [WebphoneOutgoingEventKind.CallEnd]: this.handleCallEnd,
    });
  }

  private handleEstablishedCall = (event: WebphoneEstablishedCallEvent) => {
    if (event.callKind === CallKind.Incoming) {
      this.hasActiveIncomingCall = true;

      this.acceptCall(event);
    }
  };

  private acceptCall(event: WebphoneEstablishedCallEvent) {
    jsonApiCall<AcceptCallRequest, AcceptCallResponse>({
      url: '/yacalls/acceptCall',
      data: {
        callId: event.callId,
        unitId: event.associatedUnitId,
        remoteNumber: event.remoteNumber,
        providerPhoneNumber: event.providerNumber,
      },
    }).then((data) => {
      this.acceptCallData = {
        accountId: data.accountId,
        issueTypeId: data.typeId,
        issueId: data.id,
        callId: data.callId,
      };
      establishedCallEventListener(data.callId);
      webphoneStateManager.setCrmCallId(data.callId);
      webphoneStateManager.setParentIssue(data.id, data.typeId);
      this.acceptCallSubject.next(this.acceptCallData);

      return data;
    });
  }

  private handleCallEnd = (event: WebphoneCallEndEvent) => {
    if (this.hasActiveIncomingCall && this.acceptCallData) {
      this.callEndSubject.next({
        ycCallId: event.callId,
        issueId: this.acceptCallData.issueId,
        callId: this.acceptCallData.callId,
      });

      this.acceptCallData = undefined;
      this.hasActiveIncomingCall = false;
    }
  };

  public onCallAccept(subscription: OnCallAcceptSubscription) {
    return this.acceptCallSubject.subscribe(subscription);
  }

  public onCallEnd(subscription: OnCallEndSubscription) {
    return this.callEndSubject.subscribe(subscription);
  }
}

export const webphoneIncomingCallService = new WebphoneIncomingCallService();
