import {
  CallKind,
  WebphoneOutgoingEventKind,
  WebphoneWidget,
} from '@yandex-telephony/ya-calls-webphone-sdk';
import { StateManager } from './StateManager';

export enum CallState {
  Pending = 1,
  Accepted,
}

export interface WebphoneState {
  isInit: boolean;
  call?: {
    state: CallState;
    kind: CallKind;
    id: string;
    crmCallId?: number;
  };
  issue?: {
    id: number;
    typeId: number;
  };
}

export class WebphoneStateManager extends StateManager<WebphoneState> {
  public static readonly defaultState: WebphoneState = { isInit: false };

  private webphone: WebphoneWidget;

  constructor(webphone?: WebphoneWidget) {
    super(WebphoneStateManager.defaultState);

    if (webphone) {
      this.setWebphone(webphone);
    }
  }

  public setCrmCallId(callId: number): void {
    if (this.state.call) {
      this.setState({
        call: {
          ...this.state.call,
          crmCallId: callId,
        },
      });
    }
  }

  public setParentIssue(issueId: number, issueTypeId: number): void {
    if (this.state.call) {
      this.setState({
        issue: {
          id: issueId,
          typeId: issueTypeId,
        },
      });
    }
  }

  public setWebphone(webphone: WebphoneWidget) {
    this.webphone = webphone;
    this.subscribeToWebphone();
  }

  private subscribeToWebphone() {
    this.subscribeToInit();
    this.subscribeToIncomingCall();
    this.subscribeToOutgoingCall();
    this.subscribeToCallEstablished();
    this.subscribeToCallEnd();
  }

  private subscribeToInit() {
    this.webphone.addEventListener(WebphoneOutgoingEventKind.Initialized, () => {
      this.setState({ isInit: true });
    });
  }

  private subscribeToIncomingCall() {
    this.webphone.onIncomingCall((event) => {
      this.setState({
        call: {
          ...this.state.call,
          id: event.callId,
          kind: CallKind.Incoming,
          state: CallState.Pending,
        },
      });
    });
  }

  private subscribeToOutgoingCall() {
    this.webphone.onOutgoingCall((event) => {
      this.setState({
        call: {
          ...this.state.call,
          id: event.callId,
          kind: CallKind.Outgoing,
          state: CallState.Pending,
        },
      });
    });
  }

  private subscribeToCallEstablished() {
    this.webphone.onCallEstablished((event) => {
      this.setState({
        call: {
          ...this.state.call,
          id: event.callId,
          kind: event.callKind,
          state: CallState.Accepted,
        },
      });
    });
  }

  private subscribeToCallEnd() {
    this.webphone.onCallEnd(() => {
      this.setState({ call: undefined, issue: undefined });
    });
  }
}

export const webphoneStateManager = new WebphoneStateManager();
