import { EventTarget } from 'event-target-shim';
import { XivaClient } from 'types/xiva/XivaClient';
import { createCrmXivaClient, getMultiplexHash, XivaMultiplexOptions } from 'services/XivaClient';
import { Subject, Subscription } from 'rxjs';
import { takeUntil } from 'rxjs/operators';
import {
  CRMXiva,
  BackendEvents,
  XivaBackendEvents,
  XivaBackendEventType,
  CRMXivaCustomEvent,
} from '../../types';

export class CRMXivaImpl2 extends EventTarget<XivaBackendEvents, {}> implements CRMXiva {
  private static readonly XIVA_BACKEND_EVENT_HASH = Object.values(XivaBackendEventType).reduce(
    (ac, v) => {
      ac[v] = true;
      return ac;
    },
    {},
  );

  private readonly xivaClient: XivaClient;

  private activeMultiplex = new Map<string, { count: number; subscription: Subscription }>();
  private destroy$ = new Subject<true>();

  constructor() {
    super();

    this.xivaClient = createCrmXivaClient();

    this.xivaClient
      .multiplex({ tags: 'personal' })
      .pipe(takeUntil(this.destroy$))
      .subscribe();

    this.xivaClient.messageFromServer.pipe(takeUntil(this.destroy$)).subscribe(this.handleMessage);
  }

  addTopic(topic: string): void {
    this.addMultiplex({ topic: this.tagToTopic(topic) });
  }

  removeTopic(topic: string): void {
    this.removeMultiplex({ topic: this.tagToTopic(topic) });
  }

  addTag(tags: string) {
    this.addMultiplex({ tags });
  }

  removeTag(tags: string) {
    this.removeMultiplex({ tags });
  }

  destroy() {
    this.activeMultiplex.clear();
    this.destroy$.next(true);
  }

  private addMultiplex(options: XivaMultiplexOptions) {
    const hash = getMultiplexHash(options);
    const activeMultiplexInfo = this.activeMultiplex.get(hash);

    if (activeMultiplexInfo) {
      activeMultiplexInfo.count += 1;
    } else {
      this.activeMultiplex.set(hash, {
        count: 1,
        subscription: this.xivaClient
          .multiplex(options)
          .pipe(takeUntil(this.destroy$))
          .subscribe(),
      });
    }
  }

  private removeMultiplex(options: XivaMultiplexOptions) {
    const hash = getMultiplexHash(options);
    const activeMultiplexInfo = this.activeMultiplex.get(hash);

    if (!activeMultiplexInfo) {
      return;
    }

    if (activeMultiplexInfo.count === 1) {
      activeMultiplexInfo.subscription.unsubscribe();
      this.activeMultiplex.delete(hash);
    }

    activeMultiplexInfo.count -= 1;
  }

  private tagToTopic(tag: string) {
    return tag.replace('_', '@');
  }

  // eslint-disable-next-line @typescript-eslint/no-explicit-any
  private handleMessage = (message: any) => {
    let parsedMessage = {} as BackendEvents;
    let serverNotifyId = '';
    try {
      parsedMessage = JSON.parse(message.params.data.message) as BackendEvents;
      serverNotifyId = message.params.data.server_notify_id;
    } catch (e) {
      /* empty catch */
    }

    if (CRMXivaImpl2.XIVA_BACKEND_EVENT_HASH[parsedMessage.type]) {
      const event = new CustomEvent(parsedMessage.type, {
        detail: parsedMessage.data,
      }) as CRMXivaCustomEvent;
      event.serverNotifyId = serverNotifyId;

      this.dispatchEvent(event);
    }
  };
}
