import React from 'react';
import { connect } from 'react-redux';
import PropTypes from 'prop-types';
import { Field } from 'redux-form';
import SuggestWithTags from 'lego/redux-form/SuggestWithTags';
import createI18N from '@yandex-int/i18n';
import * as keyset from './FormFields.i18n';
import filterList from '../../utils/filterArray';

const i18nMyGroup = createI18N(keyset)('myGroup');

class ManagerInput extends React.Component {
  constructor(props) {
    super(props);

    const { me, members } = props;

    const items = [{ id: me.id, login: me.login, name: me.name }, ...members];

    if (members.length) {
      items.push({ id: -1, name: i18nMyGroup, login: '' });
    }

    this.items = items;
  }

  handleGetSuggest = text => filterList(this.items, text);

  normalize = (value, prevValue) => {
    let isMyGroupSelected = false;
    if (Array.isArray(prevValue)) {
      isMyGroupSelected = !!prevValue.find(id => id === -1);
    }

    if (Array.isArray(value) && value.length) {
      const ids = value.map(item => item.id);

      if (ids.includes(-1)) {
        if (isMyGroupSelected) {
          return ids.filter(id => id !== -1);
        }
        return [-1];
      }

      return ids;
    }

    return null;
  };

  format = value => {
    if (!Array.isArray(value)) {
      return [];
    }

    return value.map(id => this.items.find(item => item.id === id)).filter(item => !!item);
  };

  render() {
    const { me, members, ...passThroughProps } = this.props;

    return (
      <Field
        component={SuggestWithTags}
        provider={this.handleGetSuggest}
        showOnFocus
        normalize={this.normalize}
        format={this.format}
        {...passThroughProps}
      />
    );
  }
}

ManagerInput.propTypes = {
  me: PropTypes.shape({
    id: PropTypes.number,
    name: PropTypes.string,
  }),
  members: PropTypes.arrayOf(
    PropTypes.shape({
      id: PropTypes.number,
      name: PropTypes.string,
    }),
  ),
};

ManagerInput.defaultProps = {
  me: {},
  members: [],
};

const mapState = state => ({
  me: state.root.info.user || {},
  members: state.root.info.teamMembers || [],
});

export default connect(mapState)(ManagerInput);
