import React, { useCallback } from 'react';
import PropTypes from 'prop-types';
import { modal as createSecondLineTicket, AppModuleString } from 'modules/requestForm2';
import { ETypeString } from 'types/entities';
import createI18N from '@yandex-int/i18n';
import CurrencySwitch from 'modules/currencySwitch/components/CurrencySwitch';
import { ModalForm as ClientFind } from 'modules/clientSelectForm';
import { ModalForm as ClientCreate } from 'modules/clientCreateForm';
import { ModalForm as ClientMove } from 'modules/clientMoveForm';
import { component as Filters } from 'pages/accounts/modules/filtersList';
import * as keyset from './Toolbar.i18n';
import Massmails from '../../containers/Massmails';
import css from './styles.modules.scss';
import Button from './Button';
import MoveInAccount from './MoveInAccount';
const i18n = createI18N(keyset);
const i18nTitle = i18n('title');
const i18nFilters = i18n('filters');
const i18nAddToMassmail = i18n('addToMassmail');
const i18nSearchAccount = i18n('searchAccount');
const i18nCreateAccount = i18n('createAccount');
const i18nRequest = i18n('request');
const i18nMoveHistory = i18n('moveHistory');

const Toolbar = ({
  onFilterChange,
  onClientSelect,
  onClientCreate,
  onFilterNew,
  onFilterSave,
  moveToAccount,
  moveToNewAccount,
  clientId,
  canMerge,
  canMoveHistory,
}) => {
  const createRequest = useCallback(() => {
    let data = { data: {}, context: { action: 'Account' } };
    if (clientId) {
      data = {
        data: { accountId: clientId, byEObject: { eid: clientId, etype: ETypeString.Account } },
        context: { module: AppModuleString.Account, action: 'Account' },
      };
    }

    createSecondLineTicket.open({
      data,
    });
  }, [clientId]);

  return (
    <div className={css.root}>
      <div className={css.title}>{i18nTitle}</div>
      <div className={css.buttons}>
        <Filters
          rootComponent={Button}
          onChange={onFilterChange}
          onSave={onFilterSave}
          onNew={onFilterNew}
        >
          {i18nFilters}
          <i className="material-icons">expand_more</i>
        </Filters>
        <Massmails rootComponent={Button}>
          {i18nAddToMassmail}
          <i className="material-icons">expand_more</i>
        </Massmails>
        <ClientFind name="SEARCH_CLIENT" onSubmit={onClientSelect}>
          {(modal, open) => (
            <Button onClick={open}>
              {i18nSearchAccount}
              {modal}
            </Button>
          )}
        </ClientFind>
        <ClientCreate name="CREATE_CLIENT" onSubmit={onClientCreate}>
          {(modal, open) => (
            <Button onClick={open}>
              {i18nCreateAccount}
              {modal}
            </Button>
          )}
        </ClientCreate>
        <Button onClick={createRequest}>{i18nRequest}</Button>
        {canMerge && (
          <MoveInAccount
            leadId={clientId}
            moveToAccount={moveToAccount}
            moveToNewAccount={moveToNewAccount}
          />
        )}
        {canMoveHistory && (
          <ClientMove name="MOVE_CLIENT" accountIdFrom={clientId}>
            {(modal, open) => (
              <Button onClick={open}>
                {i18nMoveHistory}
                {modal}
              </Button>
            )}
          </ClientMove>
        )}
      </div>
      {clientId && (
        <div className={css.breadcrumbs}>
          {i18nTitle} →{' '}
          <span className={css.breadcrumbs__highlight}>{i18n('accountId', { id: clientId })}</span>
        </div>
      )}
      <div className={css.currencySwitchWrapper}>
        <CurrencySwitch />
      </div>
    </div>
  );
};

Toolbar.defaultProps = {
  clientId: undefined,
  canMerge: false,
  canMoveHistory: false,
};

Toolbar.propTypes = {
  onFilterChange: PropTypes.func.isRequired,
  onClientSelect: PropTypes.func.isRequired,
  onClientCreate: PropTypes.func.isRequired,
  onFilterNew: PropTypes.func.isRequired,
  onFilterSave: PropTypes.func.isRequired,
  moveToAccount: PropTypes.func.isRequired,
  moveToNewAccount: PropTypes.func.isRequired,
  clientId: PropTypes.number,
  canMerge: PropTypes.bool,
  canMoveHistory: PropTypes.bool,
};

export default Toolbar;
