import React from 'react';
import PropTypes from 'prop-types';
import Access from 'utils/Access';
import cx from 'classnames';
import BlockToolbar from '../BlockToolbar';
import css from './styles.modules.scss';

const Block = (props) => {
  const {
    title,
    children,
    color,
    actions,
    style,
    toolbar: Toolbar,
    access,
    padding,
    right,
    tabs,
    withoutHeader,
    ...rest
  } = props;
  if (!Access.isRead(access)) {
    return null;
  }

  const childrenWithProps = React.Children.map(children, (child) =>
    React.isValidElement(child) ? React.cloneElement(child, { ...rest }) : child,
  );

  let toolbar = null;
  if (Toolbar) {
    toolbar = <Toolbar className={css.toolbar} {...rest} />;
  } else if (Array.isArray(tabs) && tabs.length) {
    toolbar = <BlockToolbar className={css.toolbar} tabs={tabs} {...rest} />;
  }

  return (
    <div style={style} className={css.root}>
      {withoutHeader ? (
        <div className={css.header__empty} style={{ borderTopColor: color }} />
      ) : (
        <div className={css.header} style={{ borderTopColor: color }}>
          <span className={css.title}>{title}</span>
          {toolbar}
          {right && (
            <span className={css.actions}>{typeof right === 'function' ? right() : right}</span>
          )}
          {Access.isEdit(access) && actions && (
            <span className={css.actions}>
              {typeof actions === 'function' ? actions() : actions}
            </span>
          )}
        </div>
      )}
      <div className={cx(css.body, { [css.padding]: padding })}>{childrenWithProps}</div>
    </div>
  );
};

Block.propTypes = {
  title: PropTypes.node,
  children: PropTypes.node,
  actions: PropTypes.oneOfType([PropTypes.node, PropTypes.func]),
  right: PropTypes.oneOfType([PropTypes.node, PropTypes.func]),
  toolbar: PropTypes.func,
  color: PropTypes.string,
  access: PropTypes.number,
  style: PropTypes.instanceOf(Object),
  padding: PropTypes.bool,
  tabs: PropTypes.instanceOf(Array),
  withoutHeader: PropTypes.bool,
};

Block.defaultProps = {
  title: undefined,
  children: undefined,
  actions: undefined,
  right: undefined,
  toolbar: undefined,
  color: undefined,
  access: undefined,
  style: undefined,
  padding: true,
  tabs: undefined,
  withoutHeader: undefined,
};

export default Block;
