import Highcharts from './highcharts';
import toolTip, { toolTipItem } from '../components/Tooltip';
import { currencyList } from './constants';

const day = 1000 * 60 * 60 * 24;

const prepareChartSeries = (series) => {
  const addSeries = (color, points, id) => {
    const firstSeries = series[0];
    const newSeries = {
      fillColor: color,
      pointInterval: firstSeries.pointInterval,
      pointStart: firstSeries.pointStart,
      pointEnd: firstSeries.pointEnd,
      animation: false,
      data: points,
    };

    if (typeof id !== 'undefined') {
      newSeries.id = id;
    }

    series.push(newSeries);
  };

  const areaPoints1 = [];
  const areaPoints2 = [];
  const areaPointsMin = [];

  series[0].data.forEach((item0, i) => {
    const item1 = series[1].data[i];
    const delta = item0 - item1;

    if (delta > 0) {
      areaPoints1.push(Math.abs(delta));
      areaPoints2.push(0);
    } else {
      areaPoints1.push(0);
      areaPoints2.push(Math.abs(delta));
    }

    areaPointsMin.push(item0 < item1 ? item0 : item1);
  });

  // forecast line
  let forecastIndex = -1;
  series.forEach((line, n) => {
    if (line.code === 'forecast') {
      line.pointStart += line.pointInterval * line.offset;
      line.dashStyle = 'longdash';
      line.color = '#2f7ed8';
      forecastIndex = n;
    }
  });

  // swap forecast line for saving label position
  if (forecastIndex > 0) {
    const t = series[1];
    series[1] = series[forecastIndex];
    series[forecastIndex] = t;
  }

  addSeries('#55ee55', areaPoints1);
  addSeries('#ee5555', areaPoints2);
  addSeries('rgba(0,255,255,0)', areaPointsMin, 'transparent');

  return series;
};

const chartGrowth = (props) => {
  const { data: chart, intl, targetNode, currency } = props;
  const series = [];
  const chartLabels = chart.labels;
  const timeFrom = new Date(chartLabels[0]).getTime();
  const timeTo = new Date(chartLabels[chartLabels.length - 1]).getTime();

  chart.lines.forEach((item) => {
    const markerVisible = item.points !== null && item.points.length === 1;
    const line = {
      offset: item.offset,
      code: item.code,
      name: item.name,
      data: item.points,
      pointStart: timeFrom,
      pointEnd: timeTo,
      pointInterval: day,
      type: 'line',
      animation: false,
      marker: {
        enabled: markerVisible,
      },
    };

    series.push(line);
  });

  const preparedSeries = prepareChartSeries(series);

  preparedSeries.forEach((item) => {
    item.showInLegend = false;
  });

  const currencyName = currencyList[currency.code];
  return new Highcharts.chart({
    chart: {
      renderTo: targetNode,
      type: 'area',
      height: 195,
    },
    colors: ['#0085bf', '#F7B12C'],
    credits: {
      enabled: false,
    },
    title: {
      text: chart.name,
    },
    xAxis: {
      type: 'datetime',
      tickInterval: day * 7,
      labels: {
        formatter() {
          return intl.formatDate(new Date(this.value), {
            month: 'short',
            day: 'numeric',
          });
        },
      },
    },
    yAxis: {
      minPadding: 0,
      title: {
        text: currencyName,
      },
      labels: {
        formatter() {
          return this.value / 1000.0;
        },
      },
    },
    tooltip: {
      enabled: true,
      shared: true,
      useHTML: true,
      formatter() {
        const x =
          this.x % 1 === 0
            ? intl.formatDate(new Date(this.x), {
                year: 'numeric',
                month: 'short',
                day: 'numeric',
              })
            : '';
        const data = this.points.reduceRight(
          (prevPoint, point) => prevPoint + toolTipItem(point, currencyName),
          '',
        );

        return toolTip(x, data);
      },
    },
    plotOptions: {
      area: {
        stacking: true,
        lineWidth: 0,
        shadow: false,
        marker: {
          enabled: false,
        },
        enableMouseTracking: false,
        showInLegend: false,
      },
      line: {
        zIndex: 5,
        lineWidth: 2,
        marker: {
          enabled: false,
        },
      },
    },
    series: preparedSeries,
    visible: true,
  });
};

export default chartGrowth;
