import React from 'react';
import PropTypes from 'prop-types';
import { FormattedDate } from 'react-intl';
import { connect } from 'react-redux';
import CommentForm from 'forms/Comment';
import Spinner from 'components/Spinner';
import StaffCard from 'components/StaffCard';
import PlainText from 'components/PlainText';
import FilesList from 'components/FilesList';
import Button from '@crm/components/dist/lego2/Button';
import Icon from 'lego/components/Icon';
import createI18N from '@yandex-int/i18n';
import * as commonKeyset from 'common.i18n';
import * as keyset from './ClientComment.i18n';
import { commentActionsFactory } from '../../actions';
import css from './styles.modules.scss';

const commonI18n = createI18N(commonKeyset);
const i18n = createI18N(keyset);
const i18nRemoveConfirm = i18n('removeConfirm');
const i18nEdit = commonI18n('edit');
const i18nRemove = commonI18n('remove');
const i18nSave = commonI18n('save');

const getName = (id) => `COMMENTS_${id}`;
const formIdentifier = (accountId, commentId) => `COMMENT_${getName(accountId)}_${commentId}`;

const mapDispatch = (dispatch, props) => ({
  dispatch,
  commentActions: commentActionsFactory(dispatch, props.accountId),
});

@connect(null, mapDispatch)
export default class EditableComment extends React.Component {
  constructor(props) {
    super(props);

    this.state = {
      isEdit: false,
      isLoading: false,
    };

    this.save = this.withSpinner(this.save);
    this.delete = this.withSpinner(this.delete);
  }

  toggleEdit = () => {
    const { isEdit } = this.state;

    this.setState({ isEdit: !isEdit });
    if (isEdit) {
      this.props.commentActions.refreshComments();
    }
  };

  save = ({ text }) =>
    this.props.commentActions.updateComment(this.props.data.id, text).then(() => {
      this.toggleEdit();
    });

  confirmDelete = () => {
    if (global.confirm(i18nRemoveConfirm)) {
      this.delete();
    }
  };

  delete = () => this.props.commentActions.deleteComment(this.props.data.id);

  toggleLoading = () => {
    this.setState({ isLoading: !this.state.isLoading });
  };

  withSpinner = (func) => (...args) => {
    this.toggleLoading();
    return func(...args)
      .then((res) => {
        this.toggleLoading();
        return res;
      })
      .catch((e) => {
        this.toggleLoading();
        throw e;
      });
  };

  render() {
    const { data, accountId } = this.props;

    return (
      <Spinner className={css.root} overlay visible={this.state.isLoading}>
        <div className={css.header}>
          <span className={css.left}>
            <FormattedDate
              value={new Date(this.props.data.dt)}
              weekday="long"
              year="numeric"
              month="long"
              day="numeric"
            />
          </span>
          &nbsp;
          <StaffCard user={this.props.data.author} />
          {!this.props.viewOnly && !this.state.isEdit && (
            <div className={css.controls}>
              <Button
                onClick={this.toggleEdit}
                title={i18nEdit}
                view="clear"
                size="xs"
                icon={(iconCls) => (
                  <Icon className={iconCls} pack={Icon.PACKS.MATERIAL} icon="edit" />
                )}
              />
              <Button
                onClick={this.confirmDelete}
                title={i18nRemove}
                view="clear"
                size="xs"
                icon={(iconCls) => (
                  <Icon className={iconCls} pack={Icon.PACKS.MATERIAL} icon="delete" />
                )}
              />
            </div>
          )}
        </div>
        {!this.props.viewOnly && this.state.isEdit ? (
          <CommentForm
            objId={data.id}
            form={formIdentifier(accountId, data.id)}
            onCancelClick={this.toggleEdit}
            initialValues={{ text: data.text, files: data.files }}
            onSubmit={this.save}
            okButtonLabel={i18nSave}
            filesObjectName="accountComment"
            focusOnMount
            cancelButton
          />
        ) : (
          <div className={css.body}>
            <PlainText text={data.text} link />
            {data.files && <FilesList rootClassName={css.files} files={data.files} isInline />}
          </div>
        )}
      </Spinner>
    );
  }
}

EditableComment.propTypes = {
  data: PropTypes.instanceOf(Object).isRequired,
  accountId: PropTypes.number.isRequired,
  viewOnly: PropTypes.bool,
  commentActions: PropTypes.instanceOf(Object).isRequired,
};

EditableComment.defaultProps = {
  viewOnly: false,
};
