import * as React from 'react';
import InfiniteList from 'components/InfiniteList';
import createI18N from '@yandex-int/i18n';
import * as keyset from './List.i18n';
import Contact from '../Contact';
import css from './List.module.css';

const i18n = createI18N(keyset);
const i18nName = i18n('name');
const i18nEmail = i18n('email');
const i18nPhone = i18n('phone');
const i18nNoContacts = i18n('noContacts');

interface Props {
  items: any[];
  onDelete: () => void;
  accountId: number;
  canEdit?: boolean;
  onLoad: () => void;
}

const ListView: React.FC<Props> = props => {
  const { onDelete, accountId, canEdit, children, ...other } = props;

  const renderChildren = () => {
    if (children) {
      return children;
    }

    if (Array.isArray(other.items) && other.items.length) {
      return other.items.map((item, index) => {
        const contactId = item.id || item.contactId || index;
        return (
          <Contact
            className={css.content}
            key={contactId}
            id={contactId}
            contact={item}
            onDelete={onDelete}
            accountId={accountId}
            canEdit={canEdit}
          />
        );
      });
    }

    return null;
  };

  return (
    <InfiniteList className={css.list} {...other}>
      {renderChildren()}
    </InfiniteList>
  );
};

ListView.defaultProps = {
  canEdit: false,
  accountId: undefined,
  onDelete: () => {},
};

const Preview = props => {
  const { items, onLoad, isLoading, isEof, ...rest } = props;

  const Header = (
    <div className={css.header}>
      <div className={css.name}>{i18nName}</div>
      <div className={css.email}>{i18nEmail}</div>
      <div className={css.phone}>{i18nPhone}</div>
      <div className={css.actions}>&nbsp;</div>
    </div>
  );

  return !items.length ? (
    <div className={css.root}>
      <div className={css.empty}>{i18nNoContacts}</div>
    </div>
  ) : (
    <div className={css.root}>
      <ListView
        {...rest}
        header={Header}
        items={items}
        onLoad={onLoad}
        isLoading={isLoading}
        isEof={isEof}
      />
    </div>
  );
};

Preview.defaultProps = {
  items: [],
  offset: 0,
  onLoad: () => {},
  isLoading: false,
  isEof: true,
  canEdit: false,
  accountId: undefined,
};

export default Preview;
