import React from 'react';
import PropTypes from 'prop-types';
import { Field } from 'redux-form';
import Access from 'utils/Access';
import CityInput from 'pages/accounts/components/FormFields/CityEdit';
import TerritoryInput from 'pages/accounts/components/FormFields/TerritoryInput';
import UnmanagedReasonInput from 'pages/accounts/components/FormFields/UnmanagedReasonInput';
import StaffInput from 'pages/accounts/components/FormFields/StaffInput';
import GradeInput from 'pages/accounts/components/FormFields/GradeInput';
import ContractorInput from 'pages/accounts/components/FormFields/ContractorInput';
import ContractorArrayInput from 'pages/accounts/components/FormFields/ContractorArrayInput';
import { PopupChangesRequest } from 'components/ChangesRequest';
import cx from 'classnames';
import DictionaryInput from 'pages/accounts/components/FormFields/DictionaryInput';
import TierInput from 'pages/accounts/components/FormFields/TierInput';
import BrandInput from 'pages/accounts/components/FormFields/BrandInput';
import legoReduxFormComponents from 'lego/redux-form';
import ToggleEditField from './ToggleEditField';
import PreviewInput, { preview as previewComponents } from './PreviewInput';
import Label from './Label';
import cssInput from './PreviewInput/input.modules.scss';
import FieldRow from './FieldRow';

const components = {
  CityInput,
  BrandInput,
  DictionaryInput,
  TerritoryInput,
  UnmanagedReasonInput,
  TierInput,
  ContractorInput,
  ContractorArrayInput,
  StaffInput,
  GradeInput,
};

const getEditComponent = (editComponent) => {
  if (typeof editComponent === 'string') {
    return legoReduxFormComponents[editComponent] || components[editComponent];
  }
  return editComponent;
};

const ClientInput = (props) => {
  const {
    labelText,
    labelSubText,
    value,
    preview,
    previewComponent,
    editComponent,
    previewProps,
    access,
    commits,
    commitComponent,
    editProps,
    ...other
  } = props;

  if (!Access.isRead(access)) {
    return null;
  }

  let PreviewComponent = previewComponent;
  if (typeof previewComponent === 'string' && previewComponents[previewComponent]) {
    PreviewComponent = previewComponents[previewComponent];
  }

  const isEdit = Access.isEdit(access);

  const changesRequestView = (
    <PopupChangesRequest
      items={commits}
      previewComponent={commitComponent || PreviewComponent}
      previewComponentProps={previewProps}
    />
  );
  return (
    <FieldRow>
      <Label className={cssInput.label} htmlFor={other.name} spaceAfter={!labelSubText}>
        {labelText}
        {labelSubText && <span className={cssInput.subLabel}>{labelSubText}</span>}
      </Label>
      <div className={cx(cssInput.field, { [cssInput.field_preview]: !isEdit || preview })}>
        {preview ? (
          <React.Fragment>
            <PreviewComponent value={value} {...previewProps} />
            {changesRequestView}
          </React.Fragment>
        ) : (
          <Field
            component={ToggleEditField}
            size={legoReduxFormComponents[editComponent] ? 'xs' : 's'}
            classNameInput={cssInput.input}
            previewComponent={PreviewComponent}
            editComponent={getEditComponent(editComponent)}
            previewProps={previewProps}
            isEdit={isEdit}
            afterPreviewView={changesRequestView}
            format={null}
            {...other}
          />
        )}
      </div>
    </FieldRow>
  );
};

ClientInput.defaultProps = {
  labelText: undefined,
  labelSubText: undefined,
  previewComponent: PreviewInput,
  editComponent: undefined,
  access: undefined,
  previewProps: undefined,
  preview: false,
  value: undefined,
  commits: undefined,
  commitComponent: undefined,
};

ClientInput.propTypes = {
  labelText: PropTypes.node,
  labelSubText: PropTypes.node,
  previewComponent: PropTypes.func,
  editComponent: PropTypes.oneOfType([PropTypes.func, PropTypes.string]),
  access: PropTypes.number,
  previewProps: PropTypes.instanceOf(Object),
  preview: PropTypes.bool,
  value: PropTypes.any, // eslint-disable-line react/forbid-prop-types
  commits: PropTypes.any, // eslint-disable-line react/forbid-prop-types
  commitComponent: PropTypes.any, // eslint-disable-line react/forbid-prop-types
};

export default ClientInput;
