import React, { FC, useEffect, useState } from 'react';
import { Reports as SharedReports, OrderReportTypes } from '../../components/Reports';
import { getReports } from './getReports';
import { getReportsAvailableForOrder } from './getReportsAvailableForOrder';
import { getAprilOrderToken } from './getAprilOrderToken';
import { Report } from '../../types';
import { ReportsContainerProps } from './Reports.types';

export const Reports: FC<ReportsContainerProps> = (props) => {
  const { accountId } = props;
  const [reports, setReports] = useState<Report[]>([]);
  const [reportsForOrder, setReportsForOrder] = useState<OrderReportTypes.Report[]>([]);
  const [isFetching, setIsFetching] = useState<boolean>(false);
  const [eof, setEof] = useState<boolean>(false);

  useEffect(() => {
    setIsFetching(true);
    getReports({
      accountId,
    })
      .then((response) => {
        setReports(response.data);
        setEof(response.meta.pagination.eof);
      })
      .finally(() => {
        setIsFetching(false);
      });

    getReportsAvailableForOrder().then((response) => {
      setReportsForOrder(response.data);
    });
  }, [accountId]);

  const handleLoad = () => {
    setIsFetching(true);
    getReports({
      accountId,
      offset: reports.length,
      length: 10,
    })
      .then((response) => {
        setReports(reports.concat(response.data));
        setEof(response.meta.pagination.eof);
      })
      .finally(() => {
        setIsFetching(false);
      });
  };

  const getTokenAndConcatItToUrl = (url: string) => {
    return getAprilOrderToken({
      accountId,
    }).then((response) => {
      const urlObj = new URL(url);
      urlObj.searchParams.append('crm_token', response.data.token);
      return urlObj.toString();
    });
  };

  return (
    <SharedReports
      reports={reports}
      onLoad={handleLoad}
      isLoading={isFetching}
      isEof={eof}
      reportsForOrder={reportsForOrder}
      onReportOrder={getTokenAndConcatItToUrl}
      onActionClick={getTokenAndConcatItToUrl}
    />
  );
};
