import React, { ComponentType } from 'react';
import { Field } from 'redux-form';
import Access from 'utils/Access';
import { PopupChangesRequest } from 'components/ChangesRequest';
import cx from 'classnames';
import withDictionary from 'lego/components/withDictionary';
import RsYaPriorityInput from 'pages/accounts/components/FormFields/RsYaPriorityInput';
import RsYaStatusInput from 'pages/accounts/components/FormFields/RsYaStatusInput';
import legoReduxFormComponents from 'lego/redux-form';
import Select from 'lego/redux-form/Select';
import ToggleEditField from '../ToggleEditField';
import PreviewInput, { preview as previewComponents } from '../PreviewInput';
import Label from '../Label';
import cssInput from '../PreviewInput/PreviewInput.modules.scss';
import FieldRow from '../FieldRow';
import { ClientInputProps } from './ClientInput.types';

const components = {
  RsYaPriorityInput,
  RsYaStatusInput,
  SelectWithLoad: withDictionary()(Select),
};

const getEditComponent = (
  editComponent: ComponentType<ClientInputProps> | string,
): ComponentType<ClientInputProps> => {
  if (typeof editComponent === 'string') {
    return legoReduxFormComponents[editComponent] || components[editComponent];
  }

  return editComponent;
};

const ClientInput = (props: ClientInputProps) => {
  const {
    labelText,
    labelSubText,
    value,
    preview = false,
    previewComponent = PreviewInput,
    editComponent,
    formatEditProps,
    previewProps,
    access,
    commits,
    commitComponent,
    ...other
  } = props;

  if (!Access.isRead(access)) {
    return null;
  }

  let PreviewComponent = previewComponent;
  if (typeof previewComponent === 'string' && previewComponents[previewComponent]) {
    PreviewComponent = previewComponents[previewComponent];
  }

  const isEdit = Access.isEdit(access);

  const changesRequestView = (
    <PopupChangesRequest
      items={commits}
      previewComponent={commitComponent || PreviewComponent}
      previewComponentProps={previewProps}
    />
  );

  return (
    <FieldRow>
      <Label className={cssInput.label} htmlFor={other.name} spaceAfter={!labelSubText}>
        {labelText}
        {labelSubText && <span className={cssInput.subLabel}>{labelSubText}</span>}
      </Label>
      <div className={cx(cssInput.field, { [cssInput.field_preview]: !isEdit || preview })}>
        {preview ? (
          <>
            <PreviewComponent value={value} {...previewProps} />
            {changesRequestView}
          </>
        ) : (
          <Field
            component={ToggleEditField}
            size={legoReduxFormComponents[editComponent] ? 'xs' : 's'}
            classNameInput={cssInput.input}
            previewComponent={PreviewComponent}
            editComponent={getEditComponent(editComponent)}
            formatEditProps={formatEditProps}
            previewProps={previewProps}
            isEdit={isEdit}
            afterPreviewView={changesRequestView}
            format={null}
            {...other}
          />
        )}
      </div>
    </FieldRow>
  );
};

export default ClientInput;
