import React, { Component, ComponentType } from 'react';
import forOwn from 'lodash/forOwn';
import { reduxForm, reset } from 'redux-form';
import Button from '@crm/components/dist/lego2/Button';
import Block from 'pages/accounts/modules/info/components/Block';
import Access from 'utils/Access';
import fixReinitialize from 'utils/reduxForm/fixReinitialize';
import Toolbar from 'components/Toolbar';
import { unv } from 'utils/common';
import ClientInput from '../ClientInput';
import { NESTED_VALUES, ARRAY, CATEGORY_DIRECTIONS } from './RsYaInfo.constants';
import i18nKeySet from './RsYaInfo.constants';
import { RsYaInfoProps, RsYaInfoState, AccessFields } from './RsYaInfo.types';

class RsYaInfo extends Component<RsYaInfoProps, RsYaInfoState> {
  state = { isEdit: false };

  private getAccessFields = (): AccessFields => {
    const { accessFields } = this.props;

    return Object.keys(accessFields).reduce(
      (pv, cv) => ({
        ...pv,
        [cv]: Access.and(accessFields[cv], Access.create({ read: true, edit: this.state.isEdit })),
      }),
      {},
    ) as AccessFields;
  };

  private handleSubmit = (values, dispatch, props) => {
    const { initialValues = {} } = props;

    const newValues = {};
    forOwn(values, (value, key) => {
      if (value !== initialValues[key]) {
        if (NESTED_VALUES.indexOf(key) !== -1) {
          newValues[key] = { value: (value && value.id) || undefined, updated: true };
        } else if (ARRAY.includes(key)) {
          newValues[key] = { value: (value || []).map((v) => Object(v).id), updated: true };
        } else {
          newValues[key] = { value, updated: true };
        }
      }
    });
    return props.onSubmit(newValues, values).then(() => {
      if (this) {
        this.handleToggleEdit(false);
        this.props.reload();
      }
    });
  };

  private handleToggleEdit = (flag: boolean): void => {
    const { isEdit } = this.state;

    this.setState({ isEdit: unv(flag, !isEdit) });
  };

  private handleEdit = (): void => {
    this.handleToggleEdit(true);
  };

  private handleCancel = (): void => {
    this.props.dispatch(reset(this.props.form));
    this.handleToggleEdit(false);
  };

  private extractIdFromValue = (editProps: {
    input: { value: { id: number } | number };
  }): { input: { value: number } } => {
    const value =
      typeof editProps.input.value === 'object'
        ? (editProps.input.value || {}).id
        : editProps.input.value;
    return {
      ...editProps,
      input: {
        ...editProps.input,
        value,
      },
    };
  };

  render() {
    const { handleSubmit, pristine, submitting, access, initialValues } = this.props;

    const accessFields = this.getAccessFields();

    return (
      <Block title={i18nKeySet.title1} access={access} color="#66C7FF">
        <form onSubmit={handleSubmit(this.handleSubmit)}>
          <div>
            <ClientInput
              name="rsyaSspLoginPi"
              value={initialValues.rsyaSspLoginPi}
              access={accessFields.rsyaSspLoginPi}
              editComponent="TextInput"
              type="text"
              labelText={i18nKeySet.rsyaSspLoginPi}
            />
            <ClientInput
              name="rsyaSspPageId"
              value={initialValues.rsyaSspPageId}
              access={accessFields.rsyaSspPageId}
              editComponent="TextInput"
              type="text"
              labelText={i18nKeySet.rsyaSspPageId}
            />
            <ClientInput
              name="rsyaSspBundleId"
              value={initialValues.rsyaSspBundleId}
              access={accessFields.rsyaSspBundleId}
              editComponent="TextInput"
              type="text"
              labelText={i18nKeySet.rsyaSspBundleId}
            />
            <ClientInput
              name="rsyaSspDeveloper"
              value={initialValues.rsyaSspDeveloper}
              access={accessFields.rsyaSspDeveloper}
              editComponent="TextInput"
              type="text"
              labelText={i18nKeySet.rsyaSspDeveloper}
            />
            <ClientInput
              name="rsyaDeveloperType"
              value={initialValues.rsyaDeveloperType}
              access={accessFields.rsyaDeveloperType}
              previewComponent="PreviewName"
              editComponent="SelectWithLoad"
              formatEditProps={this.extractIdFromValue}
              endpoint="/view/dicts/rsyaDeveloperType"
              type="text"
              labelText={i18nKeySet.developerType}
            />
            <ClientInput
              name="rsyaDeveloperStatus"
              value={initialValues.rsyaDeveloperStatus}
              access={accessFields.rsyaDeveloperStatus}
              previewComponent="PreviewName"
              editComponent="SelectWithLoad"
              formatEditProps={this.extractIdFromValue}
              endpoint="/view/dicts/rsyaDeveloperStatus"
              type="text"
              labelText={i18nKeySet.developerStatus}
            />
            <ClientInput
              name="rsyaSspStatus"
              labelText={i18nKeySet.appStatus}
              access={accessFields.rsyaSspStatus}
              editComponent="RsYaStatusInput"
              previewComponent="PreviewDictionary"
              size="xs"
              isSingleValue
            />
            <ClientInput
              name="rsyaSspCategory"
              previewComponent="PreviewDictionary"
              directions={CATEGORY_DIRECTIONS}
              access={accessFields.rsyaSspCategory}
              labelText={i18nKeySet.rsyaSspCategory}
              editComponent="TreeInput"
              endpoint="/view/client/rsyaSspCategory/list"
              size="xs"
            />
            <ClientInput
              name="rsyaSspPriority"
              labelText={i18nKeySet.rsyaSspPriority}
              access={accessFields.rsyaSspPriority}
              editComponent="RsYaPriorityInput"
              previewComponent="PreviewDictionary"
              size="xs"
              isSingleValue
            />
            <ClientInput
              name="rsyaSspInstallCount"
              value={initialValues.rsyaSspInstallCount}
              access={accessFields.rsyaSspInstallCount}
              editComponent="TextInput"
              type="number"
              hasClear={false}
              labelText={i18nKeySet.rsyaSspInstallCount}
            />
            <ClientInput
              name="rsyaSspDailyShowCount"
              value={initialValues.rsyaSspDailyShowCount}
              access={accessFields.rsyaSspDailyShowCount}
              editComponent="TextInput"
              type="number"
              hasClear={false}
              labelText={i18nKeySet.rsyaSspDailyShowCount}
            />
          </div>
          {Access.isEdit(access) &&
            (this.state.isEdit ? (
              <Toolbar>
                <Button type="submit" view="action" disabled={pristine || submitting} size="xs">
                  {i18nKeySet.Save}
                </Button>
                <Button view="pseudo" onClick={this.handleCancel} size="xs">
                  {i18nKeySet.Cancel}
                </Button>
              </Toolbar>
            ) : (
              <div>
                <Button view="pseudo" onClick={this.handleEdit} size="xs">
                  {i18nKeySet.Edit}
                </Button>
              </div>
            ))}
        </form>
      </Block>
    );
  }
}

// eslint-disable-next-line @typescript-eslint/no-explicit-any
export default fixReinitialize(reduxForm({})(RsYaInfo)) as ComponentType<any>;
