import React, { useContext, useMemo } from 'react';
import Resizable from 'modules/resizable';
import { DeviceType, MediaMatchContext } from 'components/MediaMatch';
import { Position, Tone } from '@crm/components/dist/Panel';
import { Route, Switch } from 'react-router-dom';
import Panel from 'modules/panel';
import NewMail from '../../containers/NewMailRouter';
import Mail from '../Mail';
import css from './Layout.module.css';
import Folders from '../Folders';

import Toolbar from '../../containers/Toolbar';
import MailsList from '../../containers/MailsList';

const SAVE_UI_KEYS = ['collapseMap'];

const configs = {
  [DeviceType.Desktop]: {
    listRouter: {
      exact: false,
      path: ['/mail'],
    },
    mails: {
      disabled: false,
      enable: 'right',
      className: undefined,
    },
  },
  [DeviceType.Tablet]: {
    listRouter: {
      exact: true,
      path: ['/mail', '/mail/folderId/:folderId?'],
    },
    mails: {
      disabled: true,
      enable: 'none',
      className: css.mails,
    },
  },
  [DeviceType.Phone]: {
    listRouter: {
      exact: true,
      path: ['/mail', '/mail/folderId/:folderId?'],
    },
    mails: {
      disabled: true,
      enable: 'none',
      className: css.mails,
    },
  },
};

const MailLayout = ({
  folderId,
  onFolderChange,
  mailId,
  onMailClick,
  onMailDblClick,
  compact,
  restoreLocation,
}) => {
  const deviceType = useContext(MediaMatchContext);

  const MemoNewMailComponent = useMemo(() => {
    return (props) => <NewMail {...props} restoreLocation={restoreLocation} />;
  }, [restoreLocation]);

  const MemoMailComponent = useMemo(() => {
    return (props) => <Mail threadIdOrMailId={props.match.params.mailId} history={props.history} />;
  }, []);

  const mainBlock = (
    <Switch>
      <Route path="/mail/new" component={MemoNewMailComponent} />
      <Route
        path={['/mail/mailId/:mailId', '/mail/folderId/:folderId/mailId/:mailId']}
        component={MemoMailComponent}
      />
      {deviceType === DeviceType.Desktop ? <Route component={MemoMailComponent} /> : null}
    </Switch>
  );

  if (compact) {
    return (
      <div className={css.root}>
        <div className={css.wrap}>{mainBlock}</div>
      </div>
    );
  }

  const config = configs[deviceType];

  return (
    <div className={css.root}>
      <div className={css.wrap}>
        <Toolbar />
        <div className={css.work}>
          <Route {...config.listRouter}>
            {({ match }) => (
              <>
                <Panel
                  position={Position.Left}
                  tone={Tone.Neowhite}
                  defaultPin
                  defaultPanelId="folders"
                  name="mail/folders"
                  minWidth={130}
                  maxWidth={400}
                  hidden={!match}
                  panels={[
                    {
                      id: 'folders',
                      title: 'Папки',
                      mountMode: 'always',
                      padding: false,
                      content: (
                        <Folders
                          classNameExtend={css.resizable}
                          folderId={folderId}
                          onClick={onFolderChange}
                          saveUI={SAVE_UI_KEYS}
                        />
                      ),
                    },
                  ]}
                />
                <Resizable
                  wrap
                  name="mail/list"
                  defaultSize={{
                    width: 320,
                  }}
                  minWidth={300}
                  maxWidth="35%"
                  hidden={!match}
                  disabled={config.mails.disabled}
                  enable={config.mails.enable}
                  className={config.mails.className}
                >
                  <MailsList
                    className={css.resizable}
                    folderId={folderId}
                    mailId={mailId}
                    onClick={onMailClick}
                    onDoubleClick={onMailDblClick}
                  />
                </Resizable>
              </>
            )}
          </Route>
          {mainBlock}
        </div>
      </div>
    </div>
  );
};

/*
MailLayout.propTypes = {
  children: PropTypes.Node,
  folderId: PropTypes.number,
  mailId: PropTypes.number,
  onFolderChange: PropTypes.func,
  onMailClick: PropTypes.func.isRequired,
  onMailDblClick: PropTypes.func.isRequired,
  compact: PropTypes.bool,
};
*/

MailLayout.defaultProps = {
  children: null,
  folderId: undefined,
  mailId: undefined,
  onFolderChange: undefined,
  compact: false,
};

export default MailLayout;
