import React from 'react';
import PropTypes from 'prop-types';
import cx from 'classnames';
import { component as FilterProvider, FilterSetter } from 'modules/filter';
import { InfiniteListView } from 'containers/InfiniteList';
import Search from '../Search';
import MailListItem from './MailListItem';
import css from './styles.modules.scss';
import { NAME, LIST_SETTINGS_NAME } from '../../constants';
import { loadMails } from '../../api';
import Empty from './MailListEmpty';
import MailListSettings from '../../containers/MailListSettings';

const onChangeSettings = (setFilter) => {
  if (onChangeSettings.setFilter !== setFilter) {
    onChangeSettings.setFilter = setFilter;
    onChangeSettings.memorize = (value) => setFilter({ value });
  }

  return onChangeSettings.memorize;
};

// eslint-disable-next-line react/prop-types
const renderSettings = ({ setFilter, filter = {} }) => (
  <MailListSettings initialValues={filter.value} onChange={onChangeSettings(setFilter)} />
);

const onKeyDown = (threadExtend) => (e, threadId) => {
  if (e.key === 'ArrowRight') {
    threadExtend(threadId);
    e.preventDefault();
  }
};

const MailsList = ({
  mailId,
  onClick,
  onDoubleClick,
  onSearch,
  filterByClient,
  resetFilter,
  threadExpand,
  className,
}) => (
  <FilterProvider name={NAME} provider={loadMails} isStore>
    {({ provider, filter }) => (
      <div
        className={cx(css.root, className)}
        role="presentation"
        onKeyUp={(e) => {
          if (e.key === 'Escape') {
            resetFilter();
          }
        }}
      >
        <Search
          className={css.search}
          initialValues={(filter && filter.key === 'search' && filter.valuesUI) || { scope: 'any' }}
          forceShowAllFilters={filter && filter.key === 'search'}
          onSubmit={onSearch}
          type="default"
          enableReinitialize
        />
        <FilterSetter
          name={NAME}
          filterKey={LIST_SETTINGS_NAME}
          render={renderSettings}
          isStore
          isActive
        />
        <InfiniteListView
          name={NAME}
          onLoad={provider}
          component={MailListItem}
          className={css.list}
          selectedId={mailId}
          onChange={onClick}
          onKeyDown={onKeyDown(threadExpand)}
          itemProps={{
            filter: filter.showFolder,
            onClick,
            onDoubleClick,
            filterByClient,
            threadExpand,
          }}
          emptyComponent={
            <Empty view={filter && filter.showFolder ? Empty.VIEW.SEARCH : Empty.VIEW.FOLDER} />
          }
          reloadOnLoadUpdate
        />
      </div>
    )}
  </FilterProvider>
);

MailsList.propTypes = {
  mailId: PropTypes.number,
  onClick: PropTypes.func.isRequired,
  onDoubleClick: PropTypes.func.isRequired,
  resetFilter: PropTypes.func.isRequired,
  onSearch: PropTypes.func,
  filterByClient: PropTypes.func,
  threadExpand: PropTypes.func.isRequired,
  className: PropTypes.string,
};

MailsList.defaultProps = {
  mailId: undefined,
  className: '',
  onSearch: () => {},
  filterByClient: () => {},
};

export default MailsList;
