import React from 'react';
import PropTypes from 'prop-types';
import { DragSource } from 'react-dnd';
import cx from 'classnames';
import Tags from 'pages/mail/components/Tags';
import Button from '@crm/components/dist/lego2/Button';
import Icon from '@crm/components/dist/lego2/Icon';
import { getTagsName } from 'pages/mail/selectors';
import { REACT_DND_KEY } from 'components/ReactDndModifiedBackend';
import { Important, Unread } from '../../modules/indicators';
import Info from './MailListItemInfo';
import css from './MailListItem.modules.scss';
import DateComponent from './DateComponent';
import mailPreviewImg from './mail_drag_preview.png';

const mailSource = {
  beginDrag(props) {
    return { id: props.id, [REACT_DND_KEY]: true };
  },
};

function collect(connect, monitor) {
  return {
    connectDragSource: connect.dragSource(),
    connectDragPreview: connect.dragPreview(),
    isDragging: monitor.isDragging(),
  };
}

class MailListItem extends React.Component {
  constructor(props) {
    super(props);
    this.handleClick = this.handleClick.bind(this);
    this.handleDoubleClick = this.handleDoubleClick.bind(this);
    this.threadIconProvider = this.threadIconProvider.bind(this);
  }

  componentDidMount() {
    const img = new Image();
    img.src = mailPreviewImg;
    img.onload = () => this.props.connectDragPreview(img, { offsetX: 0, offsetY: 0 });
  }

  handleThreadToggle = (e) => {
    e.stopPropagation();
    if (!this.props.data.isThreadFetch) {
      this.props.threadExpand({ id: this.props.data.id });
    }
  };

  handleThreadToggleDoubleClick = (e) => {
    e.stopPropagation();
  };

  handleClick(event) {
    const { onClick, data } = this.props;
    onClick(data.id, event);
  }

  handleDoubleClick(event) {
    const { onDoubleClick, data } = this.props;
    onDoubleClick(data.id, event);
  }

  threadIconProvider(iconCls) {
    const {
      data: { isThreadLoad },
    } = this.props;

    return <Icon className={iconCls} type="arrow" direction={isThreadLoad ? 'top' : 'bottom'} />;
  }

  render() {
    const {
      data,
      data: { folderName, tags, isUnread, isImportant, threadSize, isInner },
      selected,
      filter,
      filterByClient,
      connectDragSource,
    } = this.props;
    const tagsName = getTagsName(data.id);
    return connectDragSource(
      <div
        className={cx(css.root, {
          [css.root_default]: !selected,
          [css.root_selected]: selected,
          [css.root_unread]: isUnread,
          [css.root_isInner]: isInner,
        })}
        onClick={this.handleClick}
        onDoubleClick={this.handleDoubleClick}
      >
        {(filter || isInner) && folderName && <div className={css.folderName}>{folderName}</div>}
        <div className={css.notifications}>
          <Unread className={css.notificationsNew} mailId={data.id} isUnread={isUnread} />
          <Important className={css.notificationsFlag} mailId={data.id} isImportant={isImportant} />
        </div>
        <div className={css.main}>
          <div className={css.content}>
            <div className={css.row}>
              <div className={css.from} title={data.from}>
                {data.from}
              </div>
              <div className={css.date}>
                <DateComponent date={data.date} />
              </div>
            </div>
            <div className={css.row}>
              <div className={css.subject} title={data.subject}>
                {data.subject}
              </div>
              {threadSize > 1 && (
                <span className={css.threadButton}>
                  <Button
                    onClick={this.handleThreadToggle}
                    onDoubleClick={this.handleThreadToggleDoubleClick}
                    view="pseudo"
                    size="xs"
                    iconRight={this.threadIconProvider}
                  >
                    {threadSize}
                  </Button>
                </span>
              )}
            </div>
            <div className={css.preview} title={data.preview}>
              {data.preview}
            </div>
          </div>
          <Tags
            size="xxs"
            className={css.tags}
            name={tagsName}
            tags={tags}
            canEdit={false}
            destroyOnUnmount
            withRedux={Boolean(selected)}
            needInit
            asyncInit
          />
          <Info
            filterByClient={filterByClient}
            data={{
              hasFiles: data.hasFiles,
              hasReply: data.hasReply,
              hasFwd: data.hasFwd,
              hasRequest: data.requestExists,
              client: data.account,
              commentText: data.commentText,
            }}
          />
        </div>
      </div>,
    );
  }
}

MailListItem.propTypes = {
  data: PropTypes.shape({
    id: PropTypes.number,
    threadSize: PropTypes.number,
    date: PropTypes.string,
    from: PropTypes.string,
    group: PropTypes.string,
    hasFiles: PropTypes.bool,
    hasReply: PropTypes.bool,
    isInner: PropTypes.bool,
    isImportant: PropTypes.bool,
    isThreadFetch: PropTypes.bool,
    isThreadLoad: PropTypes.bool,
    isUnread: PropTypes.bool,
    preview: PropTypes.string,
    subject: PropTypes.string,
    commentText: PropTypes.string,
    tags: PropTypes.arrayOf(PropTypes.instanceOf(Object)),
  }),
  selected: PropTypes.bool,
  onClick: PropTypes.func.isRequired,
  onDoubleClick: PropTypes.func.isRequired,
  connectDragPreview: PropTypes.func.isRequired,
  connectDragSource: PropTypes.func.isRequired,
  filterByClient: PropTypes.func.isRequired,
  threadExpand: PropTypes.func.isRequired,
  filter: PropTypes.objectOf(PropTypes.any),
};

MailListItem.defaultProps = {
  data: {},
  selected: false,
  filter: undefined,
};

export default DragSource('MAIL', mailSource, collect)(MailListItem);
