import React from 'react';
import PropTypes from 'prop-types';
import cx from 'classnames';
import ToolTip from 'lego/components/ToolTip';
import createI18N from '@yandex-int/i18n';
import WordBreak from 'components/WordBreak';
import * as keyset from './MailList.i18n';
import Client from './MailListItemClient';
import css from './MailListItem.modules.scss';

const i18n = createI18N(keyset);
const i18nReplied = i18n('replied');
const i18nForwarded = i18n('forwarded');
const i18nHasAttachFiles = i18n('hasAttachFiles');
const i18nHasComment = i18n('hasComment');
const i18nHasRequest = i18n('hasRequest');

const MailListItemInfo = props => {
  const { data, filterByClient } = props;

  return (
    <div className={css.info}>
      {data.hasReply && (
        <span className={cx(css.infoItem, css.replyData)} title={i18nReplied}>
          <i className="material-icons">reply</i>
        </span>
      )}
      {data.hasFwd && (
        <span className={cx(css.infoItem, css.fwdData)} title={i18nForwarded}>
          <i className="material-icons">forward</i>
        </span>
      )}
      {data.hasFiles && (
        <span className={cx(css.infoItem, css.filesData)} title={i18nHasAttachFiles}>
          <i className="material-icons">attach_file</i>
        </span>
      )}
      {data.commentText && (
        <ToolTip
          className={cx(css.infoItem, css.commentData)}
          title={i18nHasComment}
          cls={css.commentPopup}
          tooltip={<WordBreak>{data.commentText}</WordBreak>}
          stopPropagation
          padding="xs"
        >
          <i className="material-icons">chat_bubble</i>
        </ToolTip>
      )}
      {data.hasRequest && (
        <span className={cx(css.infoItem, css.requestData)} title={i18nHasRequest}>
          <i className="material-icons">bookmark</i>
        </span>
      )}
      {data.client && (
        <Client className={css.infoItem} client={data.client} onClick={filterByClient} />
      )}
    </div>
  );
};

MailListItemInfo.propTypes = {
  data: PropTypes.shape({
    hasFiles: PropTypes.bool,
    hasReply: PropTypes.bool,
    hasFwd: PropTypes.bool,
    hasRequest: PropTypes.bool,
    client: PropTypes.shape({
      login: PropTypes.string,
      agencyLogin: PropTypes.string,
    }),
    commentText: PropTypes.string,
  }),
  filterByClient: PropTypes.func.isRequired,
};

MailListItemInfo.defaultProps = {
  data: {},
};

export default MailListItemInfo;
