import React from 'react';
import PropTypes from 'prop-types';
import { reduxForm } from 'redux-form';
import { connect } from 'react-redux';
import propsAdapter from 'hoc/propsAdapter';
import backendDataAdapter from 'containers/TreeSelect/backendDataAdapter';
import Search from './Search';
import { redux as filterToggleRedux } from './components/FilterToggle';

class SearchController extends React.Component {
  static propTypes = {
    reset: PropTypes.func.isRequired,
    onCancel: PropTypes.func,
    forceShowAllFilters: PropTypes.bool,
  };

  static defaultProps = {
    forceShowAllFilters: false,
    onCancel: undefined,
  };

  constructor(props) {
    super(props);

    this.state = {
      showFilters: false,
    };
  }

  componentDidUpdate(prevProps, prevState) {
    // сбрасываем значения фильтров при закрытии попапа
    if (
      (prevState.showFilters || prevProps.forceShowAllFilters) &&
      !(this.state.showFilters || this.props.forceShowAllFilters)
    ) {
      this.props.reset();
    }
  }

  showFilters = () => {
    this.setState({
      showFilters: true,
    });
  };

  hideFilters = () => {
    this.setState({
      showFilters: false,
    });
  };

  handleCancel = () => {
    if (typeof this.props.onCancel === 'function') {
      this.props.onCancel();
    }
    this.hideFilters();
  };

  render() {
    const visible = this.state.showFilters || this.props.forceShowAllFilters;

    return (
      <Search
        onFocusInput={this.showFilters}
        showClose={Boolean(this.props.forceShowAllFilters) && this.props.onCancel}
        showFilters={visible}
        {...this.props}
        onCancel={this.handleCancel}
        visible={visible}
        onClose={this.hideFilters}
      />
    );
  }
}

const formatSuggestUser = (data) => {
  if (Array.isArray(data)) {
    return data.map((item = {}) => item.id);
  }

  return [];
};

export default connect((state) => ({ withFilters: filterToggleRedux.selectors.getToggle(state) }))(
  propsAdapter({
    adapter: (props) => ({
      ...props,
      onSubmit: (values) => {
        let newValues;
        if (props.withFilters) {
          newValues = backendDataAdapter(values);
          if (Array.isArray(newValues.tagIds)) {
            if (newValues.tagIds.length) {
              newValues.tagIds = newValues.tagIds.map((tag) => tag.id);
            } else {
              delete newValues.tagIds;
            }
          }

          newValues.authorIds = formatSuggestUser(newValues.authorIds);
          newValues.ownerIds = formatSuggestUser(newValues.ownerIds);

          if (props.type === 'default') {
            if (newValues.folderId) {
              newValues.folderId = newValues.folderId[0]; // eslint-disable-line prefer-destructuring
            } else {
              // все папки
              newValues.folderId = -1;
            }
          }
        } else {
          newValues = { text: values.text };
        }

        return props.onSubmit(newValues, values);
      },
    }),
  })(
    reduxForm({
      form: 'mail/searchFilters',
      enableReinitialize: true,
      initialValues: {
        scope: 'any',
      },
    })(SearchController),
  ),
);
