import React from 'react';
import PropTypes from 'prop-types';
import { Field, FieldArray } from 'redux-form';
import { ETypeString } from 'types/entities';
import withDictionary from 'lego/components/withDictionary';
import SuggestStaffInPopup2 from 'lego/redux-form/SuggestStaffInPopup2';
import FieldDateRange from 'requestForms/components/FieldDateRange';
import FieldTreeSelect from 'requestForms/components/FieldTreeSelect';
import createI18N from '@yandex-int/i18n';
import * as commonKeyset from 'common.i18n';
import { TextRange } from 'lego/redux-form/TextRange';
import cx from 'classnames';
import { cnTheme } from '@yandex-lego/components/Theme';
import { crmTheme } from '@crm/components/dist/lego2/Theme/presets/themes';
import TextInputWithButton from 'lego/redux-form/TextInputWithButton';
import { TagsField as Tags, apiMail as apiTags } from 'modules/tags';
import { ModalForm as ClientModal } from 'modules/clientSelectForm';
import Select from 'lego/redux-form/Select';
import * as keyset from './Search.i18n';
import FieldWrap from './SearchField';
import css from './styles.modules.scss';
import getAccountSearchStr from '../../utils/getAccountSearchStr';
import TextInput from '../../../../lego/redux-form/TextInput';

const i18n = createI18N(keyset);
const commonI18n = createI18N(commonKeyset);
const i18nCategory = i18n('category');
const i18nAll = i18n('all');
const i18nSearchBy = i18n('searchBy');
const i18nState = i18n('state');
const i18nResolution = i18n('resolution');
const i18nType = i18n('type');
const i18nAuthor = i18n('author');
const i18nAny = i18n('any');
const i18nExecutor = i18n('executor');
const i18nQueue = i18n('queue');
const i18nLoginOrClientId = i18n('loginOrClientId');
const i18nChoose = commonI18n('choose');
const i18nInn = i18n('inn');
const i18nContractNum = i18n('contractNum');
const i18nCreation = i18n('creation');
const i18nDeadline = i18n('deadline');
const i18nCommunication = i18n('communication');
const i18nOnlyActualOwner = i18n('onlyActualOwner');
const i18nAllOwners = i18n('allOwners');

// TODO: https://st.yandex-team.ru/CRM-10735 Поддержать переводы в танкере для etype
const COMMUNICATION_OPTIONS = [
  { content: 'Комментарий', value: ETypeString.OComment },
  { content: 'Письмо', value: ETypeString.Mail },
  { content: 'Звонок', value: ETypeString.YcCall },
];

const OwnerFilterValues = {
  onlyActualOwner: '1',
  allOwners: '0',
};

const OWNER_FILTER_OPTIONS = [
  { content: i18nOnlyActualOwner, value: OwnerFilterValues.onlyActualOwner },
  { content: i18nAllOwners, value: OwnerFilterValues.allOwners },
];

const SelectWithDict = withDictionary()(Select);

class Filters extends React.Component {
  onClientSearch = (client) => {
    const { change } = this.props;
    change('clientIdLogin', getAccountSearchStr(client));
  };

  render() {
    const { className, module, fieldsAccess } = this.props;

    return (
      <div className={cx(className, cnTheme({ root: crmTheme.root }))}>
        {Boolean(fieldsAccess.category) && (
          <FieldWrap>
            <Field
              labelPrefix={i18nCategory}
              labelWidth={100}
              component={FieldTreeSelect}
              name="categoryIds"
              size="xs"
              mode="hierarchy-check"
              endpoint={`/view/support/category/list/all?module=${module}`}
              defaultValueText={i18nAll}
              props={{ name: 'categoryId' }}
            />
          </FieldWrap>
        )}
        {Boolean(fieldsAccess.scope) && (
          <FieldWrap>
            <Field
              name="scopeId"
              placeholder={i18nSearchBy}
              component={SelectWithDict}
              size="xs"
              className={css.width_full}
              componentClassName={css.width_full}
              width="max"
              defaultValueIndex={0}
              hasEmptyValue={false}
              endpoint={`/issues/searchzones?module=${module}`}
            />
          </FieldWrap>
        )}
        {Boolean(fieldsAccess.state) && (
          <FieldWrap>
            <Field
              name="stateIds"
              placeholder={i18nState}
              component={SelectWithDict}
              size="xs"
              className={css.width_full}
              componentClassName={css.width_full}
              multiple
              width="max"
              endpoint={`/issues/state?module=${module}`}
            />
          </FieldWrap>
        )}
        {Boolean(fieldsAccess.resolution) && (
          <FieldWrap>
            <Field
              name="resolutionIds"
              placeholder={i18nResolution}
              component={SelectWithDict}
              size="xs"
              className={css.width_full}
              componentClassName={css.width_full}
              multiple
              width="max"
              endpoint={`/issues/resolution?module=${module}`}
            />
          </FieldWrap>
        )}
        {Boolean(fieldsAccess.communication) && (
          <FieldWrap>
            <Field
              name="timelineETypes"
              placeholder={i18nCommunication}
              component={Select}
              size="xs"
              className={css.width_full}
              componentClassName={css.width_full}
              width="max"
              hasEmptyValue={false}
              options={COMMUNICATION_OPTIONS}
              multiple
            />
          </FieldWrap>
        )}
        {Boolean(fieldsAccess.type) && (
          <FieldWrap>
            <Field
              name="typeIds"
              placeholder={i18nType}
              component={SelectWithDict}
              size="xs"
              className={css.width_full}
              componentClassName={css.width_full}
              width="max"
              hasEmptyValue={false}
              multiple
              endpoint={`/issues/type?module=${module}`}
            />
          </FieldWrap>
        )}
        {Boolean(fieldsAccess.author) && (
          <FieldWrap>
            <Field
              labelPrefix={i18nAuthor}
              labelWidth={100}
              component={SuggestStaffInPopup2}
              variant={SuggestStaffInPopup2.StaffSuggestVariant.UsersOnly}
              module={module}
              onlyUserHadAnyRole
              name="authorIds"
              size="xs"
              defaultValueText={i18nAny}
              width="max"
            />
          </FieldWrap>
        )}
        {Boolean(fieldsAccess.owner) && (
          <FieldWrap>
            <Field
              labelPrefix={i18nExecutor}
              labelWidth={100}
              component={SuggestStaffInPopup2}
              variant={SuggestStaffInPopup2.StaffSuggestVariant.UsersOnly}
              module={module}
              onlyUserHadAnyRole
              name="ownerIds"
              size="xs"
              defaultValueText={i18nAny}
              width="max"
            />
          </FieldWrap>
        )}
        {Boolean(fieldsAccess.onlyActualOwner) && (
          <FieldWrap>
            <Field
              name="onlyActualOwner"
              component={Select}
              size="xs"
              className={css.width_full}
              componentClassName={css.width_full}
              width="max"
              hasEmptyValue={false}
              options={OWNER_FILTER_OPTIONS}
              defaultValueIndex={1}
            />
          </FieldWrap>
        )}
        {Boolean(fieldsAccess.opportunitySource) && (
          <FieldWrap>
            <Field
              name="opportunitySourceIds"
              placeholder="Источник"
              component={SelectWithDict}
              size="xs"
              className={css.width_full}
              componentClassName={css.width_full}
              width="max"
              hasEmptyValue={false}
              endpoint="/dicts/issueOpportunitySource?withHidden=true"
              multiple
            />
          </FieldWrap>
        )}
        {Boolean(fieldsAccess.queue) && (
          <FieldWrap>
            <Field
              labelPrefix={i18nQueue}
              labelWidth={100}
              component={FieldTreeSelect}
              name="queueIds"
              size="xs"
              mode="hierarchy-check"
              endpoint={`/view/support/queue/list/all?module=${module}`}
              props={{ name: 'queueId' }}
              defaultValueText={i18nAll}
              backendDataType="array"
            />
          </FieldWrap>
        )}
        {Boolean(fieldsAccess.clientIdLogin) && (
          <FieldWrap>
            <ClientModal name="SEARCH_CLIENT" onSubmit={this.onClientSearch}>
              {(modal, open) => (
                <Field
                  name="clientIdLogin"
                  className={css.width_full}
                  componentClassName={css.width_full}
                  component={TextInputWithButton}
                  placeholder={i18nLoginOrClientId}
                  onClick={open}
                  size="xs"
                  buttonLabel={i18nChoose}
                >
                  {modal}
                </Field>
              )}
            </ClientModal>
          </FieldWrap>
        )}
        {Boolean(fieldsAccess.currentAmount) && (
          <TextRange
            startName="opportunityCurrentAmountFrom"
            endName="opportunityCurrentAmountTo"
            label="Тек. сумма"
          />
        )}
        {Boolean(fieldsAccess.inn) && (
          <FieldWrap>
            <Field
              name="inn"
              component={TextInput}
              size="xs"
              className={css.width_full}
              componentClassName={css.width_full}
              placeholder={i18nInn}
            />
          </FieldWrap>
        )}
        {Boolean(fieldsAccess.contractNum) && (
          <FieldWrap>
            <Field
              name="contractNum"
              component={TextInput}
              size="xs"
              className={css.width_full}
              componentClassName={css.width_full}
              placeholder={i18nContractNum}
            />
          </FieldWrap>
        )}
        {Boolean(fieldsAccess.createDate) && (
          <FieldDateRange
            label={i18nCreation}
            className={`${css.row} ${css.row_date}`}
            fieldClassName={cx(css.date, css.date_shrink)}
            startName="from"
            endName="to"
            size="xs"
            hasIcon={false}
          />
        )}
        {Boolean(fieldsAccess.deadlineDate) && (
          <FieldDateRange
            label={i18nDeadline}
            className={`${css.row} ${css.row_date}`}
            fieldClassName={cx(css.date, css.date_shrink)}
            startName="deadlineFrom"
            endName="deadlineTo"
            size="xs"
            hasIcon={false}
          />
        )}
        {Boolean(fieldsAccess.plannedCompletionDate) && (
          <FieldDateRange
            label="План. дата"
            className={`${css.row} ${css.row_date}`}
            fieldClassName={cx(css.date, css.date_shrink)}
            startName="opportunityPlannedDateFrom"
            endName="opportunityPlannedDateTo"
            size="xs"
            hasIcon={false}
          />
        )}
        {Boolean(fieldsAccess.tag) && (
          <FieldWrap>
            <FieldArray
              name="tagIds"
              component={Tags}
              props={{
                getTags: () => apiTags.get(),
              }}
            />
          </FieldWrap>
        )}
      </div>
    );
  }
}

Filters.propTypes = {
  className: PropTypes.string,
  change: PropTypes.func.isRequired,
  module: PropTypes.string,
  fieldsAccess: PropTypes.instanceOf(Object),
};

Filters.defaultProps = {
  className: undefined,
  module: '',
  fieldsAccess: {},
};

export default Filters;
