import React from 'react';
import PropTypes from 'prop-types';
import { bindActionCreators } from 'redux';
import { history } from 'appHistory';
import { config } from 'services/Config';
import get from 'lodash/get';
import { connect } from 'react-redux';
import { AdaptiveIntervalByTabVisibility } from 'services/AdaptiveInterval';
import { setFilter, getActiveFilterKey } from 'modules/filter';
import { getState } from 'containers/TreeSelect/selectors';
import backendActions from 'api/backendActions';
import backendUpdate from 'middlewares/backendUpdate';
import browserWindow from 'utils/browserWindow';
import { folder, list, preview } from '../updaters';
import MailLayout from '../components/Layout';
import { update } from '../actions';
import getLocationPath from '../getLocationPath';
import { NAME } from '../constants';

const toInt = (val) => (val && parseInt(val, 10)) || null;
const getFolderId = (params) => toInt(get(params, 'folderId'));
const getMailId = (params) => get(params, 'mailId');

const mapState = (state) => ({
  defaultId: get(getState(state, NAME), 'default'),
  activeFilterKey: getActiveFilterKey(state, NAME),
});

const mapDispatch = (dispatch) => ({
  update: bindActionCreators(update, dispatch),
  updateList: (folderId) => {
    dispatch(
      setFilter(NAME, {
        key: 'default',
        value: { folderId },
      }),
    );
  },
  dispatch,
});

const newMailRegex = /^\/?mail\/new\/?/;

@connect(mapState, mapDispatch)
export default class MailLayoutContainer extends React.Component {
  static propTypes = {
    params: PropTypes.objectOf(PropTypes.any),
    location: PropTypes.objectOf(PropTypes.any).isRequired,
    history: PropTypes.objectOf(PropTypes.any).isRequired,
    update: PropTypes.func.isRequired,
    updateList: PropTypes.func.isRequired,
    children: PropTypes.node,
    defaultId: PropTypes.number,
    activeFilterKey: PropTypes.string,
  };

  static defaultProps = {
    params: undefined,
    defaultId: undefined,
    children: undefined,
    activeFilterKey: undefined,
  };

  constructor(props) {
    super(props);

    this.state = {
      folderId: getFolderId(props.params) || null,
      mailId: getMailId(props.params) || null,
    };
  }

  componentDidMount() {
    if (!global.COMPACT) {
      this.interval = new AdaptiveIntervalByTabVisibility(this.props.update, {
        timeoutNormal: config.value.config.intervals.viewMailListRefresh,
        timeoutSlow: config.value.config.intervals.viewMailListRefresh * 2,
      });
      backendActions.addListener('MAIL.SET_FOLDER', (payload) => {
        this.handleActionChangeFolder(payload.newFolderId);
      });
      backendUpdate.addUpdate(folder);
      backendUpdate.addUpdate(list);
    }
    backendUpdate.addUpdate(preview);
    /*
     * обновляем список писем
     * */
    if (this.state.folderId) {
      this.props.updateList(this.state.folderId);
    } else {
      this.redirectToDefaultFolderId();
    }
  }

  componentWillReceiveProps(nextProps) {
    if (!newMailRegex.test(nextProps.location.pathname)) {
      // обновление выделенной папки и письма из url
      const stateChange = {};
      let isStateChange = false;

      const nextFolderId = getFolderId(nextProps.params);
      if (nextFolderId !== this.state.folderId) {
        stateChange.folderId = nextFolderId;
        isStateChange = true;
      }

      const nextMailId = getMailId(nextProps.params);
      if (nextMailId !== this.state.mailId) {
        stateChange.mailId = nextMailId;
        isStateChange = true;
      }

      if (isStateChange) {
        this.setState(stateChange);
      }
    }
  }

  componentDidUpdate(prevProps, prevState) {
    // обновляем список писем если была изменена папка или
    // вы кликнули на туже папку
    if (!this.state.folderId) {
      this.redirectToDefaultFolderId();
    } else if (this.state.folderId !== prevState.folderId || this.folderClick) {
      this.props.updateList(this.state.folderId);
      // убираем флаг клика
      if (this.folderClick) {
        this.folderClick = false;
      }
    }
  }

  componentWillUnmount() {
    if (!global.COMPACT) {
      this.interval.destroy();
      backendActions.removeListener('MAIL.SET_FOLDER');
      backendUpdate.removeUpdate(folder);
      backendUpdate.removeUpdate(list);
    }
    backendUpdate.removeUpdate(preview);
  }

  /* получение строки pathname */
  getNextLocation = (folderId, mailId) => {
    const params = this.state;
    let nextFolderId;
    if (folderId) {
      nextFolderId = folderId;
    } else if (folderId !== false) {
      nextFolderId = params.folderId;
    }

    let nextMailId;
    if (mailId) {
      nextMailId = mailId;
    } else if (mailId !== false) {
      nextMailId = params.mailId;
    }

    return getLocationPath(nextFolderId, nextMailId);
  };

  // обработка изменения папки с бека
  handleActionChangeFolder = (folderId) => {
    this.setState({ folderId });
  };

  // клик по папке
  handleClickFolder = (folderId) => {
    this.folderClick = true;
    if (newMailRegex.test(this.props.location.pathname)) {
      this.setState({ folderId });
    } else {
      // здесь проблема
      // если следующий url будет совпадать с текущим
      // в консоле появиться warning
      // например при клике по папке для обновления списка писем
      history.push(this.getNextLocation(folderId));
    }
  };

  redirectToDefaultFolderId = () => {
    if (this.props.defaultId) {
      if (!newMailRegex.test(this.props.location.pathname)) {
        history.replace(this.getNextLocation(this.props.defaultId));
      } else {
        this.setState({ folderId: this.props.defaultId });
      }
    }
  };

  // клик по письму
  handleClickMail = (mailId, event) => {
    if (event && (event.metaKey || event.altKey)) {
      browserWindow.open(this.getNextLocation(null, mailId));
    } else if (mailId !== this.state.mailId) {
      history.push(this.getNextLocation(null, mailId));
    }
  };

  handleDblClickMail = (mailId, event) => {
    if (!(event.metaKey || event.altKey)) {
      const newWindow = browserWindow.open(
        this.getNextLocation(null, mailId),
        browserWindow.TYPES.MODAL,
      );

      newWindow.COMPACT = true;
    }
  };

  // восстановление url из this.state
  // необходимо для отмены создания нового письма
  restoreLocation = (force = false) => {
    if (force) {
      const unblock = this.props.history.block();
      unblock();
    }
    history.push(this.getNextLocation());
  };

  render() {
    const folderId = this.props.activeFilterKey === 'default' && this.state.folderId;

    return (
      <MailLayout
        folderId={folderId}
        onFolderChange={this.handleClickFolder}
        mailId={this.state.mailId}
        onMailClick={this.handleClickMail}
        onMailDblClick={this.handleDblClickMail}
        compact={global.COMPACT}
        restoreLocation={this.restoreLocation}
      >
        {React.Children.map(this.props.children, (child) =>
          React.cloneElement(child, {
            restoreLocation: this.restoreLocation,
          }),
        )}
      </MailLayout>
    );
  }
}
