import React from 'react';
import PropTypes from 'prop-types';
import { FormattedDate } from 'react-intl';
import createI18N from '@yandex-int/i18n';
import * as commonKeyset from 'common.i18n';
import Comment from 'forms/Comment/index';
import Toolbar from 'components/Toolbar';
import Button from '@crm/components/dist/lego2/Button';
import PlainText from 'components/PlainText';
import get from 'lodash/get';
import * as keyset from '../MailPreview.i18n';
import css from './styles.modules.scss';

const commonI18n = createI18N(commonKeyset);
const i18n = createI18N(keyset);
const i18nSave = commonI18n('save');
const i18nCancel = commonI18n('cancel');
const i18nComment = i18n('comment');
const i18nEdit = commonI18n('edit');
const i18nRemove = commonI18n('remove');
const i18nAddComment = i18n('addComment');

export default class MailComment extends React.Component {
  static propTypes = {
    comment: PropTypes.instanceOf(Object),
    onSave: PropTypes.func.isRequired,
    onRemove: PropTypes.func.isRequired,
    className: PropTypes.string,
    readonly: PropTypes.bool,
  };

  static defaultProps = {
    className: undefined,
    comment: undefined,
    readonly: false,
  };

  constructor(props) {
    super(props);

    this.state = {
      showNewDialog: false,
    };
  }

  showEditDialog = () => {
    this.setState({
      showNewDialog: true,
    });
  };

  hideEditDialog = () => {
    this.setState({
      showNewDialog: false,
    });
  };

  render() {
    const renderLayout = () => {
      if (this.state.showNewDialog) {
        return (
          <Comment
            form="MAIL_COMMENT"
            files={false}
            onCancelClick={this.hideEditDialog}
            initialValues={{ text: get(this, 'props.comment.text') }}
            onSubmit={(values) => this.props.onSave(values).then(() => this.hideEditDialog())}
            okButtonLabel={i18nSave}
            cancelButtonLabel={i18nCancel}
            focusOnMount
            cancelButton
          />
        );
      }

      if (this.props.comment) {
        return (
          <div>
            <div className={css.header}>
              <span className={css.title}>{i18nComment}&nbsp;</span>
              <span className={css.date}>
                (
                <FormattedDate
                  value={this.props.comment.date}
                  year="numeric"
                  month="numeric"
                  day="numeric"
                  hour="numeric"
                  minute="numeric"
                />
                )
              </span>
            </div>
            <PlainText className={css.comment} text={this.props.comment.text} link />
            {!this.props.readonly && (
              <div>
                <Toolbar>
                  <Button onClick={this.showEditDialog} size="s" view="action">
                    {i18nEdit}
                  </Button>
                  <Button onClick={this.props.onRemove} size="s" view="pseudo">
                    {i18nRemove}
                  </Button>
                </Toolbar>
              </div>
            )}
          </div>
        );
      }

      return (
        <Button onClick={this.showEditDialog} size="s" view="pseudo">
          {i18nAddComment}
        </Button>
      );
    };

    if (this.props.readonly && !this.props.comment) {
      return null;
    }

    return <div className={this.props.className}>{renderLayout()}</div>;
  }
}
